// ======================================================================
// PROVIDED CODE - DO NOT EDIT THIS FILE
// ======================================================================

#include <cstdlib>
#include <iostream>
#include <fstream>
#include "diff.h"
#include "input_output.h"


//
// A helper function that converts a string token from a Diff file
// into the EDIT enum type.
//
EDIT edit_type(const std::string &type) {
  if (type == std::string("INSERT")) { return EDIT::INSERT; }
  if (type == std::string("ERASE")) { return EDIT::ERASE; }
  if (type == std::string("REPLACE")) { return EDIT::REPLACE; }
  std::cerr << "ERROR: Invalid edit type '" << type << std::endl;
  exit(1);  
}


//
// A helper function that converts a command line string into a PARSE
// enum.
//
PARSE parse_type(const std::string &type) {
  if (type == std::string("CHAR")) { return PARSE::CHAR; }
  if (type == std::string("WORD")) { return PARSE::WORD; }
  if (type == std::string("LINE")) { return PARSE::LINE; }
  std::cerr << "ERROR: Invalid parse type '" << type << std::endl;
  exit(1);  
}


//
// This helper function escapes the double quote character, which
// allows us to create visual diffs of code with double quotes.
//
void escape_double_quotes(std::ostream &ostr, const std::string &s) {
  for (std::string::const_iterator s_itr = s.begin(); s_itr != s.end(); s_itr++) {
    if (*s_itr == '"') ostr << "\\\"";
    else ostr << *s_itr;
  }
}


//
// Helper functions to read input text (or code) files.  And write out
// text (or code) files reconstructed after applying diff edit
// operations to an input file.
//
// Input files are always stored as a list of strings.  The strings
// will be single letters (for the CHAR parse type), a WORD of one or
// more characters with no white space, or a LINE ending in a newline
// character as indicated.
//
std::list<std::string> read_text(PARSE type, const std::string &filename) {
  std::ifstream istr(filename);
  if (!istr.good()) {
    std::cerr << "ERROR: Could not open input file '" << filename << "'" << std::endl;
    exit(0);
  }

  // break the input file into strings as prescribed by the PARSE enum type
  std::string s;
  std::list<std::string> answer;
  if (type == PARSE::CHAR) {
    istr >> std::noskipws;
    char c;
    while(istr >> c) { answer.push_back(std::string(1,c)); }
    istr >> std::skipws;
  } else if (type == PARSE::WORD) {
    while(istr >> s) { answer.push_back(s); }
  } else {
    assert (type == PARSE::LINE);
    while (std::getline(istr,s)) { answer.push_back(s); }
  }
  return answer;
}

void write_text(PARSE type,
                const std::list<std::string> &text, const std::string &filename) {
  std::ofstream ostr(filename);
  assert (ostr.good());
  std::list<std::string>::const_iterator itr = text.begin();
  while (true) {
    ostr << *itr;
    itr++;
    if (itr == text.end()) break;
    if (type == PARSE::WORD) { ostr << " "; }
    if (type == PARSE::LINE) { ostr << "\n"; }
  }
  // text files should end with a newline
  if (type != PARSE::LINE) { ostr << "\n"; }
  ostr.flush();
}


//
// Helper functions to read and write difference files.  Difference
// files store the insert, erase, and replace operations.
//
// These functions may be incomplete for complicated punctuation or
// other special characters.  Students may edit this function to add
// additional escaped characters if desired to handle their additional
// testing.
//
// Note that the PARSE variable is currently not being used
//
std::list<Diff> read_diff(PARSE /*p_type*/, const std::string &filename) {
  std::ifstream istr(filename);
  if (!istr.good()) {
    std::cerr << "ERROR: Could not open diff file '" << filename << "'" << std::endl;
    exit(0);
  }

  // each row of the difference file contains a single edit operation
  int location;
  std::string type;
  std::string value;
  std::list<Diff> answer;

  while (istr >> location >> type) {
    // insert and erase operations also require a string value.  We
    // surround the value is surrounded with double quotes, in case it
    // contains whitespace.
    if (type == "INSERT" || type == "REPLACE") {
      value = "";
      char c;
      istr >> c;
      assert (c == '\"');
      istr >> std::noskipws;
      while (istr >> c) {
        if (c == '\"') break;
        if (c == '\\') {
          // allow a double quote to appear within the string, if it
          // is escaped with the backslash
          istr >> c;
          value.push_back(c);
        } else {
          value.push_back(c);
        }
      }
      istr >> std::skipws;
    }
    if (type == "INSERT") {
      answer.push_back(Diff(location,EDIT::INSERT,value));
    } else if (type == "REPLACE") {
      answer.push_back(Diff(location,EDIT::REPLACE,value));
    } else {
      assert (type == "ERASE");
      answer.push_back(Diff(location,EDIT::ERASE));
    }
  }
  return answer;
}

// Note that the PARSE variable is currently not being used
void write_diff(PARSE /*type*/, const std::list<Diff> &diff, const std::string &filename) {
  std::ofstream ostr(filename);
  assert (ostr.good());
  for (std::list<Diff>::const_iterator itr = diff.begin(); itr != diff.end(); itr++) {
    ostr << (*itr).getPosition();
    if ((*itr).getType() == EDIT::INSERT) {
      ostr << " INSERT \"";
      escape_double_quotes(ostr,(*itr).getValue());
      ostr << "\"" << std::endl;
    } else if ((*itr).getType() == EDIT::ERASE) {
      ostr << " ERASE" << std::endl;
    } else {
      ostr << " REPLACE \"";
      escape_double_quotes(ostr,(*itr).getValue());
      ostr << "\"" << std::endl;
    }
  }
}

