#include <math.h>
#include "camera.h"
#include "matrix.h"

// Included files for OpenGL Rendering
#ifdef __APPLE__
#include <OpenGL/gl.h>
#include <OpenGL/glu.h>
#include <GLUT/glut.h>
#else
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glut.h>
#endif


// ====================================================================
// ====================================================================
// CONSTRUCTORS

Camera::Camera(Vec3f c, Vec3f d, Vec3f u) {
  center = c;
  direction = d;
  up = u;
  // normalize the vectors
  up.Normalize();
  direction.Normalize();
}

OrthographicCamera::OrthographicCamera(Vec3f c, Vec3f d, Vec3f u, float s) : Camera(c,d,u) {
  size = s;
  lowerLeft = center - (getScreenUp() * (size/2.0)) - (getHorizontal() * (size/2.0));
  xAxis = getHorizontal() * size;
  yAxis = getScreenUp() * size;
}

// ====================================================================
// ====================================================================
// GL INIT
// Create an orthographic camera with the appropriate dimensions that
// crops the screen in the narrowest dimension.

void OrthographicCamera::glInit(int w, int h) {
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  float horiz = size / 2.0;
  float vert = size / 2.0;
  float aspect = float(w)/float(h);
  if (aspect > 1) 
    vert /= aspect;
  else horiz *= aspect;
  glOrtho(-horiz, horiz, -vert, vert, 0.5, 40.0);
}
 
// ====================================================================
// ====================================================================
// GL PLACE CAMERA
// Place an orthographic camera within an OpenGL scene

void Camera::glPlaceCamera(void) {
  Vec3f lookAt = center + direction;
  gluLookAt(center.x(), center.y(), center.z(),
	    lookAt.x(), lookAt.y(), lookAt.z(),
            up.x(), up.y(), up.z());
}

// ====================================================================
// zoomCamera, truckCamera, and RotateCamera
//
// Asumptions:
//  - up is really up (i.e., it hasn't been changed
//    to point to "screen up")
//  - up and direction are normalized
// Special considerations:
//  - If your constructor precomputes any vectors for
//    use in 'generateRay', you will likely to recompute those
//    values at athe end of the these three routines
// ====================================================================

// ====================================================================
// truckCamera: Translate camera perpendicular to the direction vector
// ====================================================================

void OrthographicCamera::truckCamera(float dx, float dy) {
  center += getHorizontal()*dx + getScreenUp()*dy;
  lowerLeft = center - (getScreenUp() * (size/2.0)) - (getHorizontal() * (size/2.0));
}

// ====================================================================
// rotateCamera: Rotate around the up and horizontal vectors
// ====================================================================

void OrthographicCamera::rotateCamera(float rx, float ry) {
  // Don't let the model flip upside-down (There is a singularity
  // at the poles when 'up' and 'direction' are aligned)
  float tiltAngle = acos(up.Dot3(direction));
  if (tiltAngle-ry > 3.13)
    ry = tiltAngle - 3.13;
  else if (tiltAngle-ry < 0.01)
    ry = tiltAngle - 0.01;

  Matrix rotMat = Matrix::MakeAxisRotation(up, rx);
  rotMat *= Matrix::MakeAxisRotation(getHorizontal(), ry);

  rotMat.Transform(center);
  rotMat.TransformDirection(direction);

  lowerLeft = center - (getScreenUp() * (size/2.0)) - (getHorizontal() * (size/2.0));
  xAxis = getHorizontal() * size;
  yAxis = getScreenUp() * size;
}

// ====================================================================
// zoomCamera: Make the image larger/smaller
// ====================================================================

void OrthographicCamera::zoomCamera(float factor) {
  size *= factor;
  lowerLeft = center - (getScreenUp() * (size/2.0)) - (getHorizontal() * (size/2.0));
  xAxis = getHorizontal() * size;
  yAxis = getScreenUp() * size;
}

// ====================================================================
// ====================================================================
