#ifndef MESH_H
#define MESH_H

#include <vector>
#include <string>
#include "vectors.h"
#include "hash.h"
#include "boundingbox.h"

class ArgParser;
class Vertex;
class Edge;
class Triangle;

// ======================================================================
// ======================================================================

class Mesh {

public:

  // ========================
  // CONSTRUCTOR & DESTRUCTOR
  Mesh() {}
  ~Mesh();
  void Load(const std::string &input_file);
  void ComputeGouraudNormals();
    
  // ========
  // VERTICES
  int numVertices() const { return vertices.size(); }
  Vertex* addVertex(const Vec3f &pos);
  // look up vertex by index from original .obj file
  Vertex* getVertex(int i) const {
    assert (i >= 0 && i < numVertices());
    Vertex *v = vertices[i];
    assert (v != NULL);
    return v; }

  // =====
  // EDGES
  int numEdges() const { return edges.size(); }
  // this efficiently looks for an edge with the given vertices, using a hash table
  Edge* getMeshEdge(Vertex *a, Vertex *b) const;

  // =========
  // TRIANGLES
  int numTriangles() const { return triangles.size(); }
  void addTriangle(Vertex *a, Vertex *b, Vertex *c);
  void removeTriangle(Triangle *t);

  // ===============
  // OTHER ACCESSORS
  const BoundingBox& getBoundingBox() const { return bbox; }
  Vec3f LightPosition(ArgParser *args);

  // ===============
  // OTHER FUNCTIONS
  void Paint(ArgParser *args);

private:

  // HELPER FUNCTIONS FOR PAINT
  void DrawLight(Vec3f light_position);
  void DrawMirror();
  void DrawFloor();
  void DrawMesh(ArgParser *args);
  void DrawReflectedMesh(ArgParser *args);
  void DrawReflectedFloor();
  void DrawSilhouetteEdges(Vec3f light_position);
  void DrawShadowPolygons(Vec3f light_position);

  // ==============
  // REPRESENTATION
  std::vector<Vertex*> vertices;
  edgeshashtype edges;
  triangleshashtype triangles;
  BoundingBox bbox;
};

Vec3f ComputeNormal(const Vec3f &p1, const Vec3f &p2, const Vec3f &p3);

// ======================================================================
// ======================================================================

#endif




