#include <cstdio>
#include <cassert>

#include "mesh.h"
#include "edge.h"
#include "vertex.h"
#include "triangle.h"
#include "glCanvas.h"
#include "argparser.h"

// Included files for OpenGL Rendering
#ifdef __APPLE__
#include <OpenGL/gl.h>
#else
#include <GL/gl.h>
#endif

Vec3f floor_color(0.9,0.8,0.7);
Vec3f mesh_color(1,1,1);
Vec3f mirror_color(0.1,0.1,0.2);
Vec3f mirror_tint(0.9,0.9,1.0);

// =======================================================================
// PAINT UTILITY FUNCTIONS
// =======================================================================

void InsertNormal(const Vec3f &p1, const Vec3f &p2, const Vec3f &p3) {
  Vec3f normal = ComputeNormal(p1,p2,p3);
  glNormal3f(normal.x(), normal.y(), normal.z());
}

void InsertColor(const Vec3f &v) {
  glColor3f(v.x(),v.y(),v.z());
}

// =======================================================================
// PAINT HELPER FUNCTIONS
// =======================================================================

// the light position can be animated
Vec3f Mesh::LightPosition(ArgParser *args) {
  Vec3f min = bbox.getMin();
  Vec3f max = bbox.getMax();
  Vec3f tmp;
  bbox.getCenter(tmp);
  tmp += Vec3f(0,1.5*(max.y()-min.y()),0);
  tmp += Vec3f(cos(args->timer) * (max.x()-min.x()), 0, 0);
  tmp += Vec3f(0,0,sin(args->timer) * (max.z()-min.z()));
  return tmp;
}

void Mesh::DrawLight(Vec3f light_position) {
  glPointSize(10);
  glDisable(GL_LIGHTING);
  Vec3f tmp = light_position;
  glBegin(GL_POINTS);
  glVertex3f(tmp.x(),tmp.y(),tmp.z());
  glEnd();
  glEnable(GL_LIGHTING);
}

void Mesh::DrawMirror() {
  Vec3f diff = bbox.getMax()-bbox.getMin();
  // create frame vertices just a bit bigger than the bounding box
  Vec3f a = bbox.getMin() + Vec3f(-0.25*diff.x(), 0.1*diff.y(),-0.25*diff.z());
  Vec3f b = bbox.getMin() + Vec3f(-0.25*diff.x(), 0.1*diff.y(), 1.25*diff.z());
  Vec3f c = bbox.getMin() + Vec3f(-0.25*diff.x(), 1.25*diff.y(),-0.25*diff.z());
  Vec3f d = bbox.getMin() + Vec3f(-0.25*diff.x(), 1.25*diff.y(), 1.25*diff.z());
  glBegin(GL_QUADS);
  InsertNormal(a,c,d); 
  glVertex3f(a.x(),a.y(),a.z());
  glVertex3f(c.x(),c.y(),c.z());    
  glVertex3f(d.x(),d.y(),d.z());    
  glVertex3f(b.x(),b.y(),b.z());
  glEnd();
}

void Mesh::DrawFloor() {
  Vec3f diff = bbox.getMax()-bbox.getMin();
  // create vertices just a bit bigger than the bounding box
  Vec3f a = bbox.getMin() + Vec3f(-0.75*diff.x(),0,-0.75*diff.z());
  Vec3f b = bbox.getMin() + Vec3f( 1.75*diff.x(),0,-0.75*diff.z());
  Vec3f c = bbox.getMin() + Vec3f(-0.75*diff.x(),0, 1.75*diff.z());
  Vec3f d = bbox.getMin() + Vec3f( 1.75*diff.x(),0, 1.75*diff.z());
  glBegin(GL_QUADS);
  InsertNormal(a,c,d); 
  glVertex3f(a.x(),a.y(),a.z());
  glVertex3f(c.x(),c.y(),c.z());    
  glVertex3f(d.x(),d.y(),d.z());    
  glVertex3f(b.x(),b.y(),b.z());
  glEnd();
}

// draw the original mesh
void Mesh::DrawMesh(ArgParser *args) {
  glBegin (GL_TRIANGLES);
  for (triangleshashtype::iterator iter = triangles.begin();
       iter != triangles.end(); iter++) {
    Triangle *t = iter->second;
    Vec3f a = (*t)[0]->getPos();
    Vec3f b = (*t)[1]->getPos();
    Vec3f c = (*t)[2]->getPos();    
    Vec3f na = ComputeNormal(a,b,c);
    Vec3f nb = na;
    Vec3f nc = na;
    if (args->gouraud_normals) {
      // normals averaged per vertex
      na = (*t)[0]->getGouraudNormal();
      nb = (*t)[1]->getGouraudNormal();
      nc = (*t)[2]->getGouraudNormal();
    }
    glNormal3d(na.x(),na.y(),na.z());
    glVertex3f(a.x(),a.y(),a.z());
    glNormal3f(nb.x(),nb.y(),nb.z());
    glVertex3f(b.x(),b.y(),b.z());
    glNormal3f(nc.x(),nc.y(),nc.z());
    glVertex3f(c.x(),c.y(),c.z());
  }
  glEnd();
}


// draw a second copy of the object where it appears to be on the other side of the mirror
void Mesh::DrawReflectedMesh(ArgParser *args) {


  // ASSIGNMENT: WRITE THIS FUNCTION


}


// draw a second copy of the floor where it appears to be on the other side of the mirror
void Mesh::DrawReflectedFloor() {


  // ASSIGNMENT: WRITE THIS FUNCTION


}


// figure out which edges are the silhouette of the object 
void Mesh::DrawSilhouetteEdges(Vec3f light_position) {


  // ASSIGNMENT: WRITE THIS FUNCTION


}


// project the silhouette edges away from the light source
void Mesh::DrawShadowPolygons(Vec3f light_position) {


  // ASSIGNMENT: WRITE THIS FUNCTION


}


// ======================================================================================
// ======================================================================================

void Mesh::Paint(ArgParser *args) {

  // scale it so it fits in the window
  Vec3f center; bbox.getCenter(center);
  float s = 1/bbox.maxDim();
  glScalef(s,s,s);
  glTranslatef(-center.x(),-center.y(),-center.z());

  // setup the light
  Vec3f light_position = LightPosition(args);
  GLfloat position[4] = { light_position.x(),light_position.y(),light_position.z(),1 };
  glLightfv(GL_LIGHT1, GL_POSITION, position);

  // --------------------------
  // NEITHER SHADOWS NOR MIRROR
  if (!args->mirror && !args->shadow) {
    InsertColor(mirror_color);
    DrawMirror();
    InsertColor(floor_color);
    DrawFloor();
    if (args->geometry) {
      InsertColor(mesh_color);
#ifdef __HW4_SHADERS__
      if (args->glsl_enabled) {
	glUseProgramObjectARB(GLCanvas::program);
      }
#endif
      DrawMesh(args); 
#ifdef __HW4_SHADERS__
      if (args->glsl_enabled) {
	glUseProgramObjectARB(0);
      }
#endif
    }
  } 

  // ---------------------
  // MIRROR ONLY RENDERING
  else if (args->mirror && !args->shadow) {
    // Clear frame, depth & stencil buffers
    glClearColor(1,1,1,0);
    glClearStencil(0);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
    glEnable(GL_DEPTH_TEST);

    // Draw all non-mirror geometry to frame & depth buffers
    InsertColor(mesh_color);
    DrawMesh(args);
    InsertColor(floor_color);
    DrawFloor();
    // draw back of mirror (if we can see the back)
    glEnable(GL_CULL_FACE);
    glCullFace(GL_FRONT);
    InsertColor(mirror_color);
    DrawMirror();     
    glDisable(GL_CULL_FACE);

    // Draw mirror to stencil buffer, where depth buffer passes
    glEnable(GL_STENCIL_TEST);
    glStencilOp(GL_KEEP,GL_KEEP,GL_REPLACE);
    glStencilFunc(GL_ALWAYS,1,~0); 
    // (only draw the mirror if we can see the front)
    glEnable(GL_CULL_FACE);
    glCullFace(GL_BACK);
    InsertColor(mirror_color);
    DrawMirror();     
    glDisable(GL_CULL_FACE);
    glColorMask(GL_FALSE,GL_FALSE,GL_FALSE,GL_FALSE); // disable frame buffer writes      
    glDepthRange(1,1);
    glDepthFunc(GL_ALWAYS);
    glStencilFunc(GL_EQUAL,1,~0); 
    glStencilOp(GL_KEEP,GL_KEEP,GL_KEEP);
    DrawMirror();     

    // Set depth to infinity, where stencil buffer passes
    glDepthFunc(GL_LESS);
    glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE); // enable frame buffer writes
    glDepthRange(0,1);

    // Draw reflected geometry to frame & depth buffer, where stencil buffer passes
    InsertColor(mirror_tint * mesh_color);
    DrawReflectedMesh(args);
    InsertColor(mirror_tint * floor_color);
    DrawReflectedFloor();

    glColorMask(GL_FALSE,GL_FALSE,GL_FALSE,GL_FALSE); // disable frame buffer writes      
    glStencilOp(GL_KEEP,GL_KEEP,GL_ZERO);
    glDepthFunc(GL_ALWAYS);
    DrawMirror();
    glDepthFunc(GL_LESS);
    glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE); // enable frame buffer writes    

    glDisable(GL_STENCIL_TEST);
  } 

  // ---------------------
  // SHADOW ONLY RENDERING
  else if (!args->mirror && args->shadow) {
    

    // ASSIGNMENT: WRITE THIS RENDERING MODE


  }

  // --------------------------
  // MIRROR & SHADOW!
  else {
    assert (args->mirror && args->shadow);

    // EXTRA CREDIT: IMPLEMENT THIS INTERACTION

    glColor3f(0.3,0.3,0.3);
    DrawFloor();
    DrawMesh(args);
    DrawMirror();
  }

  // -------------------------
  // ADDITIONAL VISUALIZATIONS (for debugging)
  glColor3f(1,1,0);
  DrawLight(light_position);
  if (args->reflected_geometry) {
    glColor3f(0,0,1);
    DrawReflectedMesh(args);
    DrawReflectedFloor();
  }
  if (args->bounding_box) {
    glColor3f(0,0,0);
    bbox.paint();
  }
  if (args->silhouette_edges) {
    glColor3f(1,0,0);
    DrawSilhouetteEdges(light_position);
  }
  if (args->shadow_polygons) {
    glDisable(GL_LIGHTING);
    glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
    glEnable(GL_BLEND);
    glDepthMask(GL_FALSE);
    glColor4f(0,1,0.5,0.2);
    glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
    DrawShadowPolygons(light_position);
    glDepthMask(GL_TRUE);
    glDisable(GL_BLEND);
    glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
    glEnable(GL_LIGHTING);
  }
    
  HandleGLError(); 
}

// =================================================================
