%version FINAL 41499
\documentclass{article}
\usepackage{latexsym}

\title{Graph Interface\\Proxy Components\\Using CORBA}
\author{Shawn Bisgrove \and Jan Kratky \and Cory Plock}

\begin{document}
\maketitle

\newpage

\tableofcontents
\newpage

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\section{Introduction}

This paper describes the implementation of a set of proxy components
enabling the creation of a distributed graph system using CORBA.  We
do not offer here an introduction to CORBA; some understanding of
CORBA concepts and implementation is assumed.

We describe and document only those classes not entirely generated by
an IDL compiler. For an understanding of IDL-compiler-generated class
files, see the Object Management Group's world wide web site
\cite{OMG}, or the \verb|MICO| literature at \cite{MICO}.

After a brief discussion of the concept of a ``proxy component,'' we
present the Interface Definition Language (IDL) code for our
distributed components. This section will serve as a reference to
implementors of clients that make use of these components.

Next, we document the implementation classes, after which we present
example servers and several clients that interact with these
servers. Our clients make a wide variety of calls via both the Static
Invocation Interface (SII) and the Dynamic Invocation Interface
(DII). A timing-test client provides a comparison of time taken for
local calls with time taken for distributed calls to a faster
machine. An interactive client demonstrates a node-mapping scheme that
may be useful to future client implementors.

Finally, a makefile and startup script are provided.

For implementation documentation of the Graph package used in the
below code, please see Appendix A. For documentation on
graph-algorithm implementation, see Appendix B.

\section{What is a Proxy Component?}

We use the term \verb|proxy component| to denote an object layer
between a client and a server that attempts to reformat, parse, or
convert communications.

The idea of an intermediary layer between an object and a complex
sub-system is not new.  This sort of scheme has been formalized as a
design pattern, known as the \verb|facade pattern|\cite{Gamma}, in
which a single contact point for a possibly complex system is
developed, thereby simplifying client calls to that system. The
decomposition of the problem in this fashion makes the system more
modular and, hence, more manageable.

In a distributed setting, proxy components allow calls to be made to
non-distributed code or \emph{legacy code} without ``cluttering'' that
code with the commands required for their network distribution.  The
resulting high modularity and loose coupling of system components
permits easier coding and debugging---with the use of proxies, each
component ``stands on its own'' to a greater degree.

It may be argued that adding proxy layers might slow system execution.
However, without proxies, the data types manipulated within, and
returned from, an object would be constrained to CORBA types.  In C++,
doing so limits the use of well-established specialized STL types and
algorithms having documented stability and efficiency\cite{Musser}.
Constituting an invitation to sloppy programming, such a CORBA-only
approach quite easily could lead to programmer-introduced errors and
inefficiencies.

The system's main IDL interfaces are \verb|proxy_interface| and
\verb|proxy_factory|. The latter is responsible for creating instances
of the former.  This allows a single CORBA server to generate multiple
instances of a \verb|proxy_interface| for each of the clients.
Without a construct like \verb|proxy_factory|, all connecting clients
would share the same \verb|proxy_interface| instance (not always a
desirable situation).

\section{IDL Definitions/User Reference}
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

This section contains all of the CORBA public interfaces accessible to
a client, and may be used as a reference to services offered by
our distributed proxy objects.

Four CORBA interfaces are defined within the module \verb|disgraph|
(this module name was introduced in order to resolve potential
name-scoping issues). These interfaces are \verb|proxy_interface|,
\verb|proxy_factory|, \verb|EdgeIterator| and \verb|NodeIterator|. A
client implementation will need to refer only to these interfaces in
order to access graph services and algorithms.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\subsection{IDL for the Graph Proxy}

@o graph_proxy.idl
@{module disgraph {
    @<\verb|graph_proxy.idl| typedefs specific@>
    @<\verb|graph_proxy.idl| typedefs edge@>
    @<\verb|graph_proxy.idl| typedefs lists@>
    @<\verb|graph_proxy.idl| typedefs representation@>
    @<\verb|graph_proxy.idl| exceptions@>
    @<\verb|graph_proxy.idl| iterators@>
    @<\verb|graph_proxy.idl| interface@>
    @<\verb|graph_proxy.idl| extract representation@>
    @<\verb|graph_proxy.idl| node interface@>
    @<\verb|graph_proxy.idl| edge interface@>
    @<\verb|graph_proxy.idl| algorithm object calls@>
  };
  @<\verb|graph_proxy.idl| proxy factory@>
};
@}
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

\subsubsection{Data Types}

Care has been taken to separate the data-type definitions from the
rest of the code to facilitate future redefinition. We use CORBA
built-in data types such as \verb|CORBA::String|,
\verb|CORBA::Boolean|, and \verb|CORBA::Sequence| to represent any
data that is transmitted.

@d \verb|graph_proxy.idl| typedefs specific
@{
  typedef long NodeType;
  typedef double Weight;
@}

A graph edge is defined by a triplet $<node_i, node_j, weight>$, a
compact representation of edges for the purposes of populating and
retrieving data from the graph.  An IDL \verb|struct| is used for
compound data types that are language- and platform-independent.

@d \verb|graph_proxy.idl| typedefs edge
@{  struct EdgeType {
	NodeType 	node_i;
	NodeType 	node_j;
	Weight		weight;
  };
@}

CORBA sequences allow efficient transmission of blocks of data.  Our
timing tests (presented in a later section) show that \verb|sequence|
transmission is vastly more efficient than serial transmission of
individual data items.

@d \verb|graph_proxy.idl| typedefs lists
@{  typedef sequence <EdgeType> Edges;
  typedef sequence <NodeType> Nodes;
@}

Below, the interface's representation of a graph is defined as
consisting of a \verb|nodeList| and an \verb|edgeList|.
\verb|stdGraphType| is used when transmitting data intended to
represent all edges and nodes of a graph. This more abstract
representation serves as a common representation, and is used by a
client to specify an abstraction of a graph.

@d \verb|graph_proxy.idl| typedefs representation
@{  struct stdGraphType {
	Nodes	nodeList;
	Edges	edgeList;
  };
@}


\subsubsection{Exceptions}

Below is the IDL definition of the types of exceptions that may be
raised by the \verb|graph_proxy|. However, since neither version 1.1.1
nor 1.1.2 of the Cygnus \verb|egcs| compiler appear to fully support
\verb|MICO| exception-handling, our components do not make use of the
below exceptions at this time.\footnote{Since our non-distributed
graph package does not propagate exceptions, this is of little
import.}

@d \verb|graph_proxy.idl| exceptions
@{  exception InvalidNodeType { NodeType node; };
  exception InvalidNodeList { Nodes nodeList; };
  exception InvalidEdgeType { EdgeType edge; };
  exception InvalidEdgeList { Edges edgeList; };
  exception InvalidstdGraphType { stdGraphType graph; };
  
  exception InvalidProxyInterface { };
  exception BadAllocation { stdGraphType graph; };
  exception BadFree { };
@}

\subsubsection{Iterators}

Below, we introduce the concept of an iterator within the CORBA
framework.  Such a construct can be used to iterate though a context's
binding when doing a nameservice lookup\cite{Baker}.

@d \verb|graph_proxy.idl| iterators 
@{  interface NodeIterator {
    unsigned long next_one(out NodeType node);
    unsigned long next_n (in unsigned long how_many, 
		  out Nodes nl);
    unsigned long all ( out Nodes nl );
    unsigned long length ( );
    unsigned long reset ( );
    void destroy( );
  };

  interface EdgeIterator {
    unsigned long next_one(out EdgeType edge);
    unsigned long next_n (in unsigned long how_many, 
		  out Edges el);
    unsigned long all ( out Edges el );
    unsigned long length ( );
    unsigned long reset ( );
    void destroy( );
  };
@}

\subsubsection{Proxy Interface}

Below is an IDL definition that provides a near one-to-one mapping to
methods provided by \verb|GraphInterface|. If additional
algorithms/services were to be added to the \verb|GraphInterface|,
corresponding additions would have to be made to this IDL definition
of the \verb|graph_proxy|.

The interface permits fine-grain operations on a graph; such use may
not be desirable due to the overhead incurred in the marshaling and
demarshaling process.\footnote{Such overhead may be significant---see
the timing tests presented in a later section.} A client will be
responsible for determining the level of granularity most appropriate
in its setting.

@d \verb|graph_proxy.idl| interface
@{  interface proxy_interface {
   void destroy ( )
	raises (BadFree);
@}

The allocation interface as defined by the IDL is not to be confused
with any other reference to allocation within the graph data
structures defined in Appendix A, which have to do with allocation of
internal graph structures.  These are not similar or even related.
The calls below are offered to a client after a reference to
\verb|graph_proxy| has been obtained.

@d \verb|graph_proxy.idl| extract representation
@{  stdGraphType getGraph ( )
	raises(InvalidstdGraphType);
@}

The IDL provides several methods that interact with node input types
for manipulation of a graph object:

@d \verb|graph_proxy.idl| node interface
@{  Nodes adjacentNodesOut ( in NodeType node )
	raises (InvalidNodeType);
  Nodes adjacentNodesIn ( in NodeType node )
	raises (InvalidNodeType);
  boolean isConnected ( in NodeType node ) 
	raises (InvalidNodeType);
  Weight findWeight ( in NodeType node1, in NodeType node2 )
	raises (InvalidNodeType);  
  Weight addWeights ( in Nodes nodeList )
	raises (InvalidNodeList);
  void removeOrphanNodes();
  boolean addNode ( in NodeType node )
	raises (InvalidNodeType);
  boolean addNodes ( in Nodes nodeList )
	raises (InvalidNodeList);
  boolean removeNodes ( in Nodes nodeList )
	raises (InvalidNodeList);
  Nodes getNodes ( );
  NodeIterator getINodes ( );
@}

The following methods provide for manipulation of a graph's edges:

@d \verb|graph_proxy.idl| edge interface
@{  boolean addEdge ( in EdgeType edge )
	raises (InvalidEdgeType);
  boolean addEdges ( in Edges edgeList )
	raises (InvalidEdgeList);
  boolean removeEdge ( in NodeType node_i, 
                       in NodeType node_j )
	raises (InvalidNodeType);
  Edges getEdges ( );
  EdgeIterator getIEdges ( );
@}

Finally, proxy methods for calls to algorithms that act upon graphs
are provided:

@d \verb|graph_proxy.idl| algorithm object calls
@{  boolean has_cycles ( );
  Nodes topological_sort( in NodeType u )
	raises (InvalidNodeType);
  proxy_interface prims ( in NodeType r )
	raises (InvalidNodeType);
@}

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\subsection{IDL for the Proxy Factory}

Next, we present the IDL definition for the \verb|proxy_factory|,
which handles the creation of \verb|proxy_interfaces|.

When a client connects to a server, it should get a new CORBA object
and not an object another client already is using.  Without the
functionality provided by this factory class all the clients
connecting to the system would share the same object\cite{Gamma}.

This construct also allows the client to create multiple
\verb|graph_interface| instances on the same server.

In addition to the option to \verb|create| and \verb|destroy| an
interface, a client also is given the option to \verb|clone| an
interface.

@d \verb|graph_proxy.idl| proxy factory
@{  interface proxy_factory {
    proxy_interface createEmpty ( );
    proxy_interface create ( in stdGraphType graph )
	raises (InvalidstdGraphType, BadAllocation);
    proxy_interface createEdgeNode ( in Nodes nodeList, 
                                     in Edges edgeList)
	raises (InvalidNodeList, InvalidEdgeList, BadAllocation);
    proxy_interface clone ( in proxy_interface gpi )
	raises (InvalidProxyInterface, BadAllocation);
  };
@}

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\section{Proxy Implementation Classes}

\subsection{Graph Proxy Implementation}

The graph proxy's implementation class derives from
\verb|graph_proxy_skel|, which the IDL compiler generates from the IDL
definition given in the previous section.  This is responsible for
implementing the methods and interfaces as they are defined by the
\verb|IDL|.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
@o graph_proxy_impl.h -d
@{#ifndef __GRAPH_PROXY_IMPL_H__
#define __GRAPH_PROXY_IMPL_H__

@<\verb|graph_proxy_impl| header includes@>
@<\verb|graph_proxy_impl| header class@>
@<\verb|graph_proxy_impl| node iterator class@>
@<\verb|graph_proxy_impl| edge iterator class@>

class proxy_interface_impl : 
      virtual public proxy_interface_skel
{
  @<\verb|graph_proxy_impl| interface private methods@>
  @<\verb|graph_proxy_impl| interface methods structors@>
  @<\verb|graph_proxy_impl| interface methods extract representation@>
  @<\verb|graph_proxy_impl| interface methods sequence return type@>
  @<\verb|graph_proxy_impl| interface methods boolean return type@>
  @<\verb|graph_proxy_impl| interface methods scalar return type @>
  @<\verb|graph_proxy_impl| interface methods node manipulator@>
  @<\verb|graph_proxy_impl| interface methods node retrieval@>
  @<\verb|graph_proxy_impl| interface methods edge manipulator@>
  @<Has Cycles implemented@>
  @<Topological sort implemented@>
  @<Prims algorithm implemented@>
};
@<\verb|graph_proxy_impl| factory class@>

#endif
@}
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

The \verb|graph_proxy_impl| class provides the implementation of the
proxy components defined in the IDL.  Its methods map STL types to
CORBA-defined types and vice-versa, implementing a full set of proxy
components.

@d \verb|graph_proxy_impl| header includes
@{#include "graph_proxy.h"  
#include "matrixgraph.h"
#include "adjacencygraph.h"
#include "algorithms.h"
@}

Below, the non-distributed graph type handled by this server is
determined by a preprocessor definition passed in as a command-line
option.

@d \verb|graph_proxy_impl| header class 
@{using namespace disgraph;

#ifdef MAT
typedef matrix_graph<NodeType, NodeType> GraphType;
#else
typedef adjacency_graph<NodeType, NodeType> GraphType;
#endif
@}

\subsubsection{Iterators}

Below, we implement two types of iterators to allow the user to
iterate through return types. These are not iterators as defined by
STL, but are useful for incremental extraction of data in a
distributed setting\cite{Baker}. A client's memory may be limited;
such iteration will enable a client to use minimal memory for
examining returned data.

\subsubsection{Node Iteration}

The \verb|NodeIterator_impl| class provides for incremental retrieval
of node data.

@d \verb|graph_proxy_impl| node iterator class 
@{class NodeIterator_impl : 
        virtual public NodeIterator_skel {
private:
  CORBA::ULong idx;
  Nodes nlist;

public:
  NodeIterator_impl( const Nodes& nl ): idx(0), nlist(nl) {}  
  virtual ~NodeIterator_impl ( ) { 
	nlist.length(0);
  };
  @<node \verb|next_one|@>
  @<node \verb|next_n|@>
  @<node \verb|all|@>
  @<node \verb|length|@>
  @<node \verb|reset|@>
  @<node \verb|destroy|@>
};
@} 

The below method, \verb|next_one|, retrieves the next node and
advances the iterator.

@d node \verb|next_one|
@{  CORBA::ULong 
  NodeIterator_impl::next_one ( NodeType& node ) {
    if (idx < nlist.length() ) {
      node = nlist[idx++];
      return 1;
    } else return 0;      
  }
@}
 
The \verb|next_n| method retrieves a passed-in number of nodes, and
advances the iterator accordingly.

@d node \verb|next_n|
@{
  CORBA::ULong
  NodeIterator_impl::next_n( CORBA::ULong how_many, 
	   		     Nodes*& nl ) {

    CORBA::Long left = nlist.length() - idx;
    CORBA::ULong num_ret = 
	( ( left >= how_many ) ? how_many : left );
    nl = new Nodes(num_ret);
    nl->length (num_ret);
    for (CORBA::ULong i = 0; i < num_ret; ++i, ++idx)
      (*nl)[i] = nlist[idx];
    return num_ret;
  }
@}

The \verb|all| method returns all nodes in the graph, regardless of
the current iterator position. The iterator position remains
unchanged.

@d node \verb|all|
@{  CORBA::ULong 
  NodeIterator_impl::all ( Nodes*& nl ) {
      CORBA::ULong num_ret = nlist.length();
      nl = new Nodes(num_ret); 
      nl->length(num_ret);
      for (CORBA::ULong i = 0; i < num_ret; ++i)
	 (*nl)[i] = nlist[i];
      return num_ret;
   }
@}     

The below method, \verb|length|, yields the number of nodes in the graph.

@d node \verb|length|
@{ CORBA::ULong NodeIterator_impl::length ( ) {
	return nlist.length();
  }
@}

\verb|reset| returns the iterator position to the beginning of the
node list:

@d node \verb|reset|
@{ CORBA::ULong NodeIterator_impl::reset ( ) {
	return (idx = 0);
  }
@}

Finally, \verb|destroy| deletes the iterator and reclaims its memory.

@d node \verb|destroy|
@{ void NodeIterator_impl::destroy ( ) {
	delete this;	
  }
@}

\subsubsection{Edge Iteration}

Below is an implementation of an Edge Iterator for incremental
examination of a graph's edge data.  The functionality of each of its
proxy methods is analogous to that of the corresponding
\verb|NodeIterator| methods.

@d \verb|graph_proxy_impl| edge iterator class 
@{class EdgeIterator_impl : 
      virtual public EdgeIterator_skel {
private:
  CORBA::ULong idx;
  Edges elist;

public:
  EdgeIterator_impl( const Edges& el )
	: idx(0), elist(el) { }
  virtual ~EdgeIterator_impl ( ) {
	elist.length(0);
  };
  @<edge \verb|next_one|@>
  @<edge \verb|next_n|@>
  @<edge \verb|all|@>
  @<edge \verb|length|@>
  @<edge \verb|reset|@>
  @<edge \verb|destroy|@>
};
@}

@d edge \verb|next_one|
@{  CORBA::ULong 
  EdgeIterator_impl::next_one ( EdgeType& edge ) {
     if (idx < elist.length() ) {
	edge = elist[idx++];
        return 1;
     } else return 0;
  }
@}
  
@d edge \verb|next_n|
@{  CORBA::ULong
  EdgeIterator_impl::next_n( CORBA::ULong how_many, 
	   		        Edges*& el ) {
    CORBA::Long left = elist.length() - idx;
    CORBA::ULong num_ret = 
	( ( left >= how_many ) ? how_many : left );
    el = new Edges(num_ret);
    el->length (num_ret);
    for (CORBA::ULong i = 0; i < num_ret; ++i, ++idx)
      (*el)[i] = elist[idx];
    return num_ret;
  }
@}

@d edge \verb|all|
@{  CORBA::ULong 
  EdgeIterator_impl::all ( Edges*& el ) {
      CORBA::ULong num_ret = elist.length();
      el = new Edges(num_ret);
      el->length(num_ret);
      for (CORBA::ULong i = 0; i < num_ret; ++i)
	 (*el)[i] = elist[i];
      return num_ret;
   }
@}     

@d edge \verb|length|
@{ CORBA::ULong 
  EdgeIterator_impl::length ( ) {
	return elist.length();
  }
@}

@d edge \verb|reset|
@{ CORBA::ULong
  EdgeIterator_impl::reset ( ) {
	return (idx = 0);
  }
@}

@d edge \verb|destroy|
@{ void
  EdgeIterator_impl::destroy ( ) {
	delete this;	
  }
@}

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\subsubsection{Interface Methods}

A client of this server should not be constrained to making calls to a
particular graph representation, as long as the graph's public
interface is what the client expects.

In our system, a server can be created specifically for each type of
graph representation. When a client contacts the naming service, that
client can request a particular representation.

The server is responsible for the construction of a
\verb|GraphInterface| object. On the other hand, the client is
responsible for the selection of the appropriate internal
representation and population of the representation, as well as its
eventual deletion.

@d \verb|graph_proxy_impl| interface private ...
@{  private:	
    GraphType* dg;

    has_cycle<NodeType, NodeType>* cycle;  	
    topo_sort<NodeType, NodeType>* topsort;     
    mst_prim<NodeType, NodeType>* mst;          
@}

We now present the proxy methods. One major concern is the error
checking that the proxy must do to appropriately propagate an error
over the CORBA connection. To address this concern, a standard set of
exceptions has been defined in the IDL specification of the
\verb|graph_proxy| interface. However, due to lack of \verb|egcs|
compiler support (\verb|MICO| exceptions often are never actually
thrown) at the time of this writing, these were not implemented.

The client is responsible for maintaining any mapping between node
numbers in the distributed graph and the original node data and labels
(which may be of any type). We will, in a later section, present an
interactive client that does exactly this.

\subsubsection{Construction and Destruction}

The constructor method for the \verb|graph_proxy_impl| class contains
no operations.

However, the deconstructor method should make sure that allocated
memory, if any, is reclaimed from the graph. However, a client is
responsible for making sure that this destruction is called explicitly
when the memory is no longer required. If a client exits without
calling the appropriate \verb|destroy| method, the server does not
reclaim memory. \verb|Arno Puder|, co-creator of \verb|MICO|, in an
email discussion, confirmed that such explicit destruction calls
\emph{must} be made: ``(The destructor) will not be called implicitly
when there is no more reference pointing to the object. CORBA does not
do a global garbage collection. However, if you dispose the object
explicitly, then of course the destructor will be
called.''\footnote{Notice that this lack of garbage collection means
that if a client crashes or for other unexpected reasons fails to
destroy an object, significant server-side memory leaks may occur.}

For this purpose, \verb|graph_proxy_impl| provides a \verb|destroy|
method a client must call explicitly for destruction of a server
object. As Bjarne Stroustrup, creator of C++, has stated, ``Explicit
calls of destructors ... should be avoided whenever possible.
Occasionally, they are essential.''\cite{Stroustrup} We have here a
case where the use of explicit destruction is unavoidable.

@d \verb|graph_proxy_impl| interface methods structors
@{  public:
    proxy_interface_impl ( ) { 
      dg = new GraphType( true );

      cycle = new has_cycle<NodeType, NodeType>();  
      topsort = new topo_sort<NodeType, NodeType>();    
      mst = new mst_prim<NodeType, NodeType>();         
    }

    virtual ~proxy_interface_impl ( ) {
	if (dg != NULL) delete( dg );
	if (cycle != NULL) delete( cycle );
	if (topsort != NULL) delete( topsort );
	if (mst != NULL) delete( mst );
    } 

    void destroy ( )
      throw(::CORBA::SystemException_var,
        BadFree_var)
    { delete this; }	
@} 

\subsubsection{Data Extraction and Manipulation}

@d \verb|graph_proxy_impl| interface methods extract representation
@{    stdGraphType* getGraph() 
      throw(
        ::CORBA::SystemException_var,
        InvalidstdGraphType_var)
     {
	stdGraphType* retval = new stdGraphType;
	Nodes* nl = getNodes();
	Edges* el = getEdges();
	retval->edgeList = *el;
	retval->nodeList = *nl;
	delete nl;  // need to clean up
	delete el;  // need to clean up
	return retval;
     }
@}

In choosing algorithms and services to implement, we took care to
choose ones such that many different styles of proxies could be
demonstrated.

The below methods demonstrate the conversion to a
\verb|CORBA::Sequence| from a \verb|vector| return.  Performing this
conversion in a single process space is more efficient than using an
iterator over the network\cite{Baker}; to minimize network traffic the
entire \verb|CORBA::Sequence| is returned.

@d \verb|graph_proxy_impl| interface methods sequence ...  
@{    Nodes* adjacentNodesOut( NodeType node ) 
      throw(::CORBA::SystemException_var,
        InvalidNodeType_var)
      {
	vector<NodeType> nl;
	dg->adjacentNodesOut(node, back_inserter(nl));
  
	Nodes* retval = new Nodes(nl.size());
	retval->length(nl.size());
	for (CORBA::ULong i = 0; i < nl.size(); ++i) 
	    (*retval)[i] = nl[i];
  	return retval; 
      }

    Nodes* adjacentNodesIn( NodeType node )
      throw(::CORBA::SystemException_var,
        InvalidNodeType_var)
      {
	vector<NodeType> nl;
	dg->adjacentNodesIn(node, back_inserter(nl));
  
	Nodes* retval = new Nodes(nl.size());
	retval->length(nl.size());
	for (CORBA::ULong i = 0; i < nl.size(); ++i) 
	  (*retval)[i] = nl[i];
  	return retval; 
     }
@}

The below method demonstrates the \verb|Boolean| return capabilities
of CORBA and the simplicity of creating a proxy methods once the
framework is in place.  Calls from client to server are simply passed
on to the \verb|GraphInterface|; since this occurs in a single process
space, very little overhead is incurred.

@d\verb|graph_proxy_impl| interface methods boolean ...  
@{   CORBA::Boolean isConnected( NodeType node )
     throw (::CORBA::SystemException_var,
       InvalidNodeType_var)
    {
	return dg->isConnected ( node );
    }
@}

The below proxy method demonstrates the return of a scalar value.

@d \verb|graph_proxy_impl| interface methods scalar ...
@{    Weight addWeights( const Nodes& nodeList ) 
      throw(::CORBA::SystemException_var,
        InvalidNodeType_var)
     {
	vector<NodeType> conv(nodeList.length());
	for (CORBA::ULong i = 0; i < nodeList.length(); ++i)
	  conv[i] = nodeList[i];
	return dg->addWeights(conv.begin(), conv.end());
     }
@}

The below node-manipulation methods convert the \verb|in| parameter
types (if any) to types acceptable to the \verb|graph_interface|
package.  The newly-constructed object, if so needed, is then passed
on.

@d \verb|graph_proxy_impl| interface methods node manipu...
@{    void removeOrphanNodes()
    {
	dg->removeOrphanNodes();
    }

    CORBA::Boolean addNode( NodeType node )
      throw(::CORBA::SystemException_var,
        InvalidNodeType_var)
    {
	dg->addNode( node );
	return true;
    }

    CORBA::Boolean addNodes( const Nodes& nodeList ) 
      throw(::CORBA::SystemException_var,
        InvalidNodeList_var)
    {
	for (CORBA::ULong i = 0; i < nodeList.length(); ++i) 
	  dg->addNode( nodeList[i] );
	return true;	
    }

    CORBA::Boolean removeNodes( const Nodes& nodeList )
      throw(::CORBA::SystemException_var,
        InvalidNodeList_var)
    {
	vector<NodeType> conv(nodeList.length());
	for (CORBA::ULong i = 0; i < nodeList.length(); ++i)
	  conv[i] = nodeList[i];
	dg->removeNodes(conv.begin(), conv.end());
	return true;
    }
@}

@d \verb|graph_proxy_impl| interface methods node retrieval...
@{    Nodes* getNodes() {
	vector<NodeType> nl;
	copy(dg->begin(), dg->end(), back_inserter(nl));
	Nodes* retval = new Nodes(nl.size());
	retval->length(nl.size());
	for (CORBA::ULong i = 0; i < nl.size(); ++i)
	  (*retval)[i] = nl[i];
  	return retval; 
    }

    NodeIterator_ptr getINodes() {
	vector<NodeType> nl;
	copy(dg->begin(), dg->end(), back_inserter(nl));
	Nodes retval;
	retval.length(nl.size());
	for (CORBA::ULong i = 0; i < nl.size(); ++i)
	  retval[i] = nl[i];
	NodeIterator_ptr ni = new NodeIterator_impl( retval );
        return NodeIterator::_duplicate (ni);
    }
@}

By now, a well-established pattern of extraction and conversion has
been established for the creation of proxy methods. The below methods
use this paradigm to extract edge information:

@d\verb|graph_proxy_impl| interface methods edge manipu...
@{    @<\verb|gpi| Adding and removing edges@>
    @<\verb|gpi findWeight|@>
    @<\verb|gpi getEdges|@>
    @<\verb|gpi getIEdges|@>
@}

@d \verb|gpi findWeight|
@{    Weight findWeight ( NodeType node1, NodeType node2 )
      throw(
        ::CORBA::SystemException_var,
        InvalidNodeList_var)
    {
	return dg->findWeight( node1, node2 );
    }
@}

@d \verb|gpi getEdges|
@{    Edges* getEdges() {
	typedef pair<NodeType, NodeType> OneEdge;
	vector<OneEdge> severalEdges;

	dg->getEdges( back_inserter( severalEdges ) );

	vector<Weight> theWeights;
	for( CORBA::ULong i = 0; i < severalEdges.size(); ++i )
	  theWeights.push_back( dg->findWeight( 
            severalEdges[i].first, severalEdges[i].second ) );

	Edges* retval = new Edges(severalEdges.size());
	retval->length(severalEdges.size());
	for( CORBA::ULong i = 0; i < severalEdges.size(); ++i) {
	   ((*retval)[i]).node_i = severalEdges[i].first;	  
	   ((*retval)[i]).node_j = severalEdges[i].second;	  
	   ((*retval)[i]).weight = theWeights[i];
  	}
  	return retval; 
     }
@}

@d \verb|gpi getIEdges|
@{    EdgeIterator_ptr getIEdges() {
	typedef pair<NodeType, NodeType> OneEdge;
	vector<OneEdge> severalEdges;

	dg->getEdges( back_inserter( severalEdges ) );

	vector<Weight> theWeights;
	for( CORBA::ULong i = 0; i < severalEdges.size(); ++i )
	  theWeights.push_back( dg->findWeight( 
            severalEdges[i].first, severalEdges[i].second ) );

	Edges retval;
	retval.length(severalEdges.size());
	for( CORBA::ULong i = 0; i < severalEdges.size(); ++i) {
	   retval[i].node_i = severalEdges[i].first;	  
	   retval[i].node_j = severalEdges[i].second;	  
	   retval[i].weight = theWeights[i];
  	}

	EdgeIterator_ptr ei = new EdgeIterator_impl( retval );
        return EdgeIterator::_duplicate (ei);
    }
@}

The below methods provide for addition and removal of edges:

@d \verb|gpi| Adding and removing edges
@{    CORBA::Boolean addEdge( const EdgeType& edge )
      throw(::CORBA::SystemException_var,
        InvalidEdgeType_var)
    {
	return dg->addEdge(edge.node_i, edge.node_j,
			   edge.weight);
    }

    CORBA::Boolean addEdges( const Edges& edgeList )
      throw(::CORBA::SystemException_var,
        InvalidEdgeList_var)
    {
	for (CORBA::ULong i = 0; i < edgeList.length();++i)
	  if( !addEdge(edgeList[i]) )
            return false;
	return true;
    }

    CORBA::Boolean
    removeEdge( NodeType node_i, NodeType node_j)
      throw(::CORBA::SystemException_var,
        disgraph::InvalidNodeType_var)
    {
	return dg->removeEdge(node_i, node_j);
    }
@}

\subsubsection{Algorithm Calls}

The below algorithm calls each invoke the \verb|run| method of the
appropriate algorithm class (in \verb|algorithms.h|; see Appendix B).

@d Has Cycles implemented
@{    CORBA::Boolean has_cycles ( ) {
        return cycle->run( *dg );
    }
@}

The \verb|topological_sort| algorithm returns a sequence of nodes; the
below proxy method converts the return from \verb|run| into a form
suitable for transmission.

@d Topological sort implemented
@{    Nodes* topological_sort( NodeType u )
    throw(
      ::CORBA::SystemException_var,
      InvalidNodeType_var)
   {
      vector<NodeType> nodesBack;
      topsort->run( *dg, u, back_inserter( nodesBack ) );

      Nodes* retval = new Nodes(nodesBack.size());
      retval->length( nodesBack.size() );
      for( CORBA::ULong i = 0; i < nodesBack.size(); ++i ) 
        (*retval)[i] = nodesBack[i];

      return retval; 
    }
@}

Our implementation of Prim's minimum spanning tree algorithm returns a
handle to a \verb|proxy_interface| object. The below proxy method
therefore must do a bit more work than the previous two in order to
convert the data returned from \verb|run| into a form suitable for
CORBA transport.

@d Prims algorithm implemented
@{  virtual proxy_interface_ptr prims ( NodeType r )
   throw( ::CORBA::SystemException_var,
    InvalidNodeType_var)
  {
    GraphType returnGraph(true);
    returnGraph = mst->run( *dg, r );

    vector<NodeType> nodesBack;
    copy( returnGraph.begin(), returnGraph.end(), 
          back_inserter( nodesBack ) );

    Nodes* returnNodes = new Nodes(nodesBack.size());
    returnNodes->length(nodesBack.size());
    for( CORBA::ULong i = 0; i < nodesBack.size(); ++i) 
      (*returnNodes)[i] = nodesBack[i];

    typedef pair<NodeType, NodeType> OneEdge;
    vector<OneEdge> severalEdges;
    returnGraph.getEdges( back_inserter( severalEdges ) );

    vector<Weight> theWeights;
    for( CORBA::ULong i = 0; i < severalEdges.size(); ++i )
      theWeights.push_back( 
        returnGraph.findWeight( severalEdges[i].first,
	                        severalEdges[i].second ) );

    vector<EdgeType> edgesBack;
    Edges* returnEdges = new Edges(severalEdges.size());
    returnEdges->length(severalEdges.size());
    for( CORBA::ULong i = 0; i < severalEdges.size(); ++i) {
      ((*returnEdges)[i]).node_i = severalEdges[i].first;	  
      ((*returnEdges)[i]).node_j = severalEdges[i].second;	  
      ((*returnEdges)[i]).weight = theWeights[i];
    }

    proxy_interface_ptr ngpi = new proxy_interface_impl;
    
    ngpi->addNodes( *returnNodes );
    ngpi->addEdges( *returnEdges );

    delete returnNodes;  //need to clean up
    delete returnEdges;  //need to clean up

    return proxy_interface::_duplicate( ngpi ); 
  }
@}

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\subsection{Proxy Factory Implementation}

The proxy factory is responsible for the creation of
\verb|graph_interface| objects.

@d \verb|graph_proxy_impl| factory class 
@{class proxy_factory_impl : 
	virtual public proxy_factory_skel
{
 public:
  proxy_factory_impl ( ) {} 
  virtual ~proxy_factory_impl ( ) {} 

  @<factory \verb|create|@>
  @<factory \verb|createEdgeNode|@>
  @<factory \verb|createEmpty|@>
  @<factory \verb|clone|@>
};
@}

Given an abstract representation of a graph (\verb|stdGraphType|), 
a client can create a proxy interface.

@d factory \verb|create|
@{  virtual proxy_interface_ptr 
    create( const stdGraphType& graph ) 
      throw(
        ::CORBA::SystemException_var,
        InvalidstdGraphType_var,
        BadAllocation_var)
 {
    proxy_interface_ptr gpi  = 
	new proxy_interface_impl;
    gpi->addNodes( graph.nodeList );
    gpi->addEdges( graph.edgeList );
    return proxy_interface::_duplicate (gpi);
  }
@}

A client may also create a proxy interface by passing a list of nodes 
and a list of edges.

@d factory \verb|createEdgeNode|
@{  virtual proxy_interface_ptr 
    createEdgeNode( const Nodes& nodeList, 
                    const Edges& edgeList ) 
  throw( ::CORBA::SystemException_var,
    InvalidNodeList_var,
    InvalidEdgeList_var,
    BadAllocation_var)
  {
    proxy_interface_ptr gpi  = 
	new proxy_interface_impl;
    gpi->addNodes( nodeList );
    gpi->addEdges( edgeList );
    return proxy_interface::_duplicate (gpi);
  }
@}

An empty proxy interface may also be requested:

@d factory \verb|createEmpty|
@{  virtual proxy_interface_ptr createEmpty ( ) {
    proxy_interface_ptr gpi  = 
	new proxy_interface_impl;
    return proxy_interface::_duplicate (gpi);
  }
@}

The \verb|clone| method allows a client to copy an existing interface:

@d factory \verb|clone|
@{  virtual proxy_interface_ptr clone 
    ( proxy_interface_ptr gpi) 
    throw( ::CORBA::SystemException_var,
      InvalidProxyInterface_var,
      BadAllocation_var)
  {
    proxy_interface_ptr ngpi  = 
	new proxy_interface_impl;
    Nodes * nl = gpi->getNodes();
    Edges * el = gpi->getEdges();

    ngpi->addNodes( *nl );
    ngpi->addEdges( *el );

    delete nl;  //need to clean up 
    delete el;  //need to clean up
    return proxy_interface::_duplicate (ngpi);
  }
@}

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\section{Example Servers and Clients}

This section presents code for the \verb|server| and \verb|client|.
Currently, the client constructs both static (SII) calls as well as
DII calls.  Although computationally costly, DII calls are more
flexible than SII calls.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\subsection{Server}
@o server_ns.cc -d 
@{@<\verb|server_ns| includes@>
int main( int argc, char *argv[] ) {
  @<\verb|server_ns| orb setup@>
  @<\verb|server_ns| nameservices@>
  @<\verb|server_ns| run orb@>
}
@}
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

This is a simple server but it is robust enough to demonstrate the
major features of the component implementation.  An end user will
decide on which mode this server will run.  In our examples, the
server is run in shared mode\cite{Baker}.

@d \verb|server_ns| includes
@{#include <CORBA.h>
#include <mico/naming.h>
#include <iostream.h>
#include "graph_proxy.h"
#include "graph_proxy_impl.h"

using namespace disgraph;
@}

The below code is used to set up the \verb|MICO| ORB. Then, a
\verb|proxy_factory| is created---a client can use this to create a
\verb|proxy_interface| for access to the non-distributed graph classes
(Appendix A).

@d \verb|server_ns| orb setup
@{  CORBA::ORB_var orb = 
	CORBA::ORB_init( argc, argv, "mico-local-orb" );
  CORBA::BOA_var boa = 
	orb->BOA_init (argc, argv, "mico-local-boa");

  cerr << "[Initialization of the server complete]" << endl;

  proxy_factory_ptr 
	myGraphProxyFactory = new proxy_factory_impl();
@}

Below, the server object contacts the naming services and registers
itself.

@d \verb|server_ns| nameservices
@{    @<sns contact naming service@>
    @<sns register with naming service@>
@}

@d sns contact naming service
@{    CosNaming::NamingContext_var nc;
    try {
      CORBA::Object_var nsobj = 
	 orb->resolve_initial_references ("NameService");
 
      nc = CosNaming::NamingContext::_narrow (nsobj);
      if ( CORBA::is_nil (nc) ) {
	cerr << "Failed to narrow naming context" << endl;
	exit(-1);
      } 
    }
    catch(CORBA::ORB::InvalidName& ex) {
      cerr << "Service required is invalid [does not exist]." 
           << endl;
      exit(-1);
    }
    catch(...) {
      cerr << "Unknown exception thrown at resolving" << endl;
      exit(-1);
    }
@}

Below, the server object registers with the naming service. When the
\verb|bind| fails (this will occur if the requested name is already
bound to another object), a call to \verb|rebind| causes the naming
service to register the current object, overriding any existing
references.

@d sns register with naming service
@{    CosNaming::Name name;
    try {
      name.length (1);
      name[0].id = CORBA::string_dup ("graph_proxy_factory");

#ifdef MAT
      name[0].kind = CORBA::string_dup ("matrix");
      cerr << "[Matrix Server Registering]" << endl;
#else
      name[0].kind = CORBA::string_dup ("adjacency");
      cerr << "[Adjacency Server Registering]" << endl;
#endif

      try {
        nc->bind (name, myGraphProxyFactory);
      }
      catch(CosNaming::NamingContext::AlreadyBound& ex) {
        nc->rebind (name, myGraphProxyFactory);
      }
      catch(...) {
        nc->rebind (name, myGraphProxyFactory);
      }
      cerr << "[Server Name Registered]" << endl;
    }
    catch (CORBA::COMM_FAILURE& ex) {
      cerr << "Caught system exception COMM_FAILURE, unable to"
	 << " contact the naming service." << endl;
      exit(-1);
    }
    catch (...) {
      cerr << "Caught unknown exception while"
	 << " contacting the naming service." << endl;
      exit(-1);
    }
@}

Once the naming service has been notified, the server is ready to
begin serving the \verb|proxy_factory| instance to the various
clients.

@d \verb|server_ns| run orb
@{   boa->impl_is_ready ( CORBA::ImplementationDef::_nil( ));
   orb->run ();
   return 0;
@}


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\subsection{Client}

The sample client presented in this subsection demonstrates a wide
range of the functionality offered by our system.

@o client_ns.cc -d 
@{@<\verb|client| includes@>
@<\verb|client| helper functions@>
int main ( int argc, char * argv[]) {
  @<\verb|client| orb setup@>
  @<\verb|client| nameservices@>
  @<\verb|client| SII call allocation@>
  @<\verb|client| SII call first ...@>
  @<\verb|client| SII call algorithms@>
  @<\verb|client| SII call iterator@>
  @<\verb|client| SII call second ...@>
  @<\verb|client| SII call edge ...@>
  @<\verb|client| DII call node list return@>
  @<\verb|client| DII call scalar return@>
}
@}

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

@d \verb|client| includes
@{#include <CORBA.h>
#include "graph_proxy.h"
#include <iostream.h>
#include <mico/naming.h>

using namespace disgraph;
@}

A handful of functions are provided to simplify interface calls:

@d \verb|client| helper functions
@{@<\verb|putNode|@>
@<\verb|putEdge|@>
@<\verb|printEdges|@>
@<\verb|printNodes|@>
@}

@d \verb|putNode|
@{NodeType putNode( NodeType x ) {
  return x;
}
@}

@d \verb|putEdge|
@{
EdgeType putEdge( NodeType x, NodeType y,
              Weight w ) {
  EdgeType anEdge;
  anEdge.node_i = x;
  anEdge.node_j = y;
  anEdge.weight = w;
  return anEdge;
}
@}

@d \verb|printEdges|
@{void printEdges( Edges * edgesBack ) {
  cout << "EDGE LISTING :" << endl;
  for( int i = 0; i < (*edgesBack).length(); ++i ) {
    cout << ((*edgesBack)[i]).node_i << " ";
    cout << ((*edgesBack)[i]).node_j << " ";
    cout << ((*edgesBack)[i]).weight << endl;
  }
}
@}

@d \verb|printNodes|
@{void printNodes( Nodes * nodesBack ) {
  cout << "NODE LISTING :" << endl;
  for( int i = 0; i < (*nodesBack).length(); ++i )
    cout << (*nodesBack)[i] << endl;
}
@}

Below, the \verb|MICO| ORB is initialized.

@d \verb|client| orb setup
@{  Nodes nodes_list;
  NodeType node_num;

  CORBA::ORB_var orb = 
	CORBA::ORB_init( argc, argv, "mico-local-orb" );
@} 

Below, the client gets its naming context, and then requests a
reference to a \verb|proxy_factory| instance created by the server.

@d \verb|client| nameservices 
@{  @<Client get naming context@>
  @<Client request proxy factory@>
@}

@d Client get naming context
@{  CosNaming::NamingContext_var nc;
  try {
    CORBA::Object_var nsobj = 
      orb->resolve_initial_references ("NameService");

    nc = CosNaming::NamingContext::_narrow (nsobj);
    if ( CORBA::is_nil (nc) ) {
      cerr << "Failed to narrow naming context" << endl;
      exit(-1);
    }
  }

  catch(CORBA::ORB::InvalidName& ex) {
    cerr << "Service required is invalid [does not exist]." 
         << endl;
    exit(-1);
  }

  catch(...) {
    cerr << "Unknown exception thrown at resolving" << endl;
    exit(-1);
  }
@}

Below, the client chooses, at run-time, the type of graph server it
will access:

@d Client request proxy factory
@{  CosNaming::Name name;
  CORBA::Object_var obj;
  proxy_factory_var fdg;
  int input;

  try {
    name.length (1);
    name[0].id = CORBA::string_dup ("graph_proxy_factory");
    cout << "What type of g server would you like (0=adj/1=mat)? ";
    cin >> input;
    if (input == 0) 
       name[0].kind = CORBA::string_dup ("adjacency");
    else 
       name[0].kind = CORBA::string_dup ("matrix");

    obj = nc->resolve (name);
    fdg = proxy_factory::_narrow( obj );
  }
  catch (CORBA::COMM_FAILURE& ex) {
     cerr << "Caught system exception COMM_FAILURE, unable to"
          << " contact the naming service." << endl;
     exit(-1);
  }
  catch (...) {
     cerr << "Caught unknown exception while"
         << " contacting the naming service." << endl;
     exit(-1);
  }
@}

Note that the type \verb|proxy_interface_var| is required rather than
the type \verb|proxy_interface_ptr|.  The distinction between the two
is quite simple, but invariably, when a mistake is made, client
results are somewhat unpredictable.

CORBA defines two types of variables that are available to the
programmer. The first, \verb|_ptr|, is probably the most flexible.  When
this one is used it is the responsibility of the programmer to
manipulate the reference count and call \verb|CORBA::_release| when
done using it.  The second, \verb|_var|, automatically takes care of the
manipulation of the reference count as to not burden the user.  It is
highly recommended, when writing a client, to use \verb|_var| unless
other flexibility is needed.  The omission of proper manipulation of
the count causes unpredictable results and may cause the system to
fail. Use of \verb|CORBA::_release| on a \verb|_var| will produce
incorrect results.

In the below code, the user is prompted for the number of nodes in a
graph to be allocated. Such a graph is then allocated using SII.
 
@d \verb|client| SII call allocation
@{  int size;
  cout << "Enter the size: ";
  cin >> size;     

  proxy_interface_var dg; 

  cout << "Calling init";
  cout << "Testing allocation" << endl;
  stdGraphType* graph = new stdGraphType;
  (graph->nodeList).length(size);
  for (int j = 1; j <= size; j++) 
    (graph->nodeList)[j-1] = j;
  
  (graph->edgeList).length(size);
  for (int count = 2; count <= size; count++) {
	  (graph->edgeList)[count-2].node_i = 1;
	  (graph->edgeList)[count-2].node_j = count;
	  (graph->edgeList)[count-2].weight = 3000;
  }

  dg = fdg->create( (*graph) );
  cout << "Allocated Graph" << endl;
@}

Below, a variety of SII calls are made.

@d \verb|client| SII call first run
@{  Nodes* nodeList = dg->adjacentNodesOut( 1 );
  cout << "Call Adjacent Nodes Out complete" << endl; 
  cout << "Printing out return: " << nodeList->length() << endl;
  printNodes( nodeList );

  cout << "Trying getNodes..." << endl;
  nodeList = dg->getNodes();
  cout << "Call Get Nodes complete" << endl; 
  cout << "Printing out return: " << nodeList->length() << endl;
  printNodes( nodeList );

  node_num = 1;  
  nodes_list.length(1);
  nodes_list[0] = 4;  
  dg->removeNodes( nodes_list );
  cout << "Call Remove Nodes complete" << endl;

  cout << "Running Adjacent Nodes Out again" << endl;
  nodeList = dg->adjacentNodesOut( node_num );
  cout << "Call Adjacent Nodes Out complete" << endl; 
  cout << "Printing out return: " << nodeList->length() << endl;
  printNodes( nodeList );

  nodeList = dg->adjacentNodesIn( node_num );
  cout << "Call Adjacent Nodes In complete" << endl; 
  cout << "Printing out return: " << nodeList->length() << endl;
  printNodes( nodeList );
@}

@d \verb|client| SII call algorithms
@{@<\verb|has_cycles| test@>
@<\verb|topological_sort| test@>
@<\verb|min_st| test@>
@}

Below, a pair of calls to \verb|has_cycles| are made on a very simple
graph.  The first call should produce false; with the addition of an
edge to the graph, a cycle is created---this is reflected in the
return from the second call to \verb|has_cycles|.

@d \verb|has_cycles| test
@{  cout << "Calling has_cycle (should be false)..." << endl;
  proxy_interface_var dg2 = fdg->createEmpty();

  dg2->addNode ( putNode( 0 ));
  dg2->addNode ( putNode( 1 ));
  dg2->addNode ( putNode( 2 ));
  
  dg2->addEdge ( putEdge( 0, 1, 0 ) );
  dg2->addEdge ( putEdge( 1, 2, 0 ) );

  if( dg2->has_cycles() )
  cout << "Has a cycle";
  else cout << "No cycle!";
  cout << endl;

  dg2->addEdge ( putEdge( 2, 0, 0 ) );

  if( dg2->has_cycles() )
    cout << "Has a cycle";
  else cout << "No cycle!";
  cout << endl;

  printEdges( dg2->getEdges() ); 
@}

Below, a simple three-layer graph undergoes a topological sort:

@d \verb|topological_sort| test
@{  dg2->removeEdge( 1, 2 );
  dg2->removeEdge( 2, 0 ); // remove cycle

  dg2->addNode ( putNode( 3 ) );
  dg2->addEdge ( putEdge( 0, 2, 0 ) );
  dg2->addEdge ( putEdge( 1, 3, 0 ) );
  dg2->addEdge ( putEdge( 2, 3, 0 ) );

  // 3-layer graph ihas been created ...
  printEdges( dg2->getEdges() );

  Nodes* nodesBack;
  nodesBack = dg2->topological_sort( 0 );
  cout << "Nodes returned from topological sort: " << endl;
  printNodes ( nodesBack );
@}

Finally, \verb|prims| is tested on a graph of moderate size. This
example is culled from \cite{Cormen}, p.508.

@d \verb|min_st| test
@{  cout << "Working on mst..." << endl;
  proxy_interface_var dg3 = fdg->createEmpty();

  cout << "Placing nodes ... " << endl;
  for( int i = 0; i < 9; ++i ) 
    dg3->addNode ( putNode( i ) );

  cout << "Placing edges ... " << endl;
  dg3->addEdge ( putEdge(  0, 1, 4 ) );
  dg3->addEdge ( putEdge(  1, 2, 7 ) );
  dg3->addEdge ( putEdge(  2, 3, 7 ) );
  dg3->addEdge ( putEdge(  3, 4, 9 ) );
  dg3->addEdge ( putEdge(  4, 5, 10 ) );
  dg3->addEdge ( putEdge(  5, 6, 2 ) );
  dg3->addEdge ( putEdge(  6, 7, 1 ) );
  dg3->addEdge ( putEdge(  7, 0, 8 ) );
  dg3->addEdge ( putEdge(  1, 7, 11 ) );
  dg3->addEdge ( putEdge(  7, 8, 7 ) );
  dg3->addEdge ( putEdge(  2, 8, 2 ) );
  dg3->addEdge ( putEdge(  2, 5, 4 ) );
  dg3->addEdge ( putEdge(  3, 5, 14 ) );
  dg3->addEdge ( putEdge(  6, 8, 6 ) );

  cout << "Calling prims ... " << endl;
  proxy_interface_var dg4;// = fdg->createEmpty();
  dg4 = dg3->prims( 0 );

  cout << "Printing edges ... " << endl;
  printEdges( dg4->getEdges() ); 
@}

The below calls demonstrate use of the edge and node iterators:

@d \verb|client| SII call iterator
@{  cout << "Testing Iterator" << endl;
  NodeIterator_var temp;
  NodeType it;
  Nodes* itList;
  cout << "------------ getting one ---------" << endl;
  temp = dg->getINodes();
  if ( temp->next_one(it) ) 
    cout << it << endl;
  else cout << "Returned nothing" << endl;

  cout << "------------ getting five ---------" << endl;
  temp->next_n(5, itList);
  for (CORBA::ULong i = 0; i < itList->length(); ++i) 
	cout << (*itList)[i] << endl;

  cout << "------------ getting five ---------" << endl;
  temp->next_n(5, itList);
  for (CORBA::ULong i = 0; i < itList->length(); ++i) 
	cout << (*itList)[i] << endl;

  cout << "------------ getting all ---------" << endl;
  temp->all(itList);
  for (CORBA::ULong i = 0; i < itList->length(); ++i) 
        cout << (*itList)[i] << endl;

  cout << "------------ getting reset ---------" << endl;
  temp->reset();
  cout << "------------ getting five ---------" << endl;
  temp->next_n(5, itList);
  for (CORBA::ULong i = 0; i < itList->length(); ++i) 
        cout << (*itList)[i] << endl;

  cout << "End Iterator test" << endl;
@}

Next, the use of edge iterators is demonstrated. The results are
compared those of an initial call to \verb|getEdges|.

@d \verb|client| SII call edge stuff 
@{
  @<SII \verb|getEdges| call@>
  @<SII Edge Iterator Tests misc.@>
  @<SII Edge Iterator reset@>
  cout << "done with serial" << endl; 
@}

@d SII \verb|getEdges| call
@{  Edges* edgeList;
  cout << "Trying getEdges..." << endl;
  edgeList = dg->getEdges();
  cout << "Call Get Edge complete" << endl; 
  cout << "Printing out return: " << edgeList->length() << endl;
  printEdges( edgeList );
@}

@d SII Edge Iterator Tests misc.
@{  cout << "Testing Edge Iterator" << endl;
  EdgeIterator_var e_temp;
  EdgeType e_it;
  Edges* e_itList;
  cout << "------------ getting one ---------" << endl;
  e_temp = dg->getIEdges();
  if ( e_temp->next_one(e_it) ) 
     cout << "(" << e_it.node_i << "," << e_it.node_j <<
	 "," << e_it.weight << ")" << endl;
  else cout << "Returned nothing" << endl;

  cout << "------------ getting five ---------" << endl;
  e_temp->next_n(5, e_itList);
  for (CORBA::ULong i = 0; i < e_itList->length(); ++i) {
	cout << "(" << (*e_itList)[i].node_i << "," << 
		(*e_itList)[i].node_j << "," << 
		(*e_itList)[i].weight << ")" << endl;
  }
  cout << "------------ getting five ---------" << endl;
  e_temp->next_n(5, e_itList);
  for (CORBA::ULong i = 0; i < e_itList->length(); ++i) {
	cout << "(" << (*e_itList)[i].node_i << "," << 
		(*e_itList)[i].node_j << "," << 
		(*e_itList)[i].weight << ")" << endl;
  }
  cout << "------------ getting all ---------" << endl;
  e_temp->all(e_itList);
  for (CORBA::ULong i = 0; i < e_itList->length(); ++i) {
        cout << "(" << (*e_itList)[i].node_i << "," <<
                (*e_itList)[i].node_j << "," <<
                (*e_itList)[i].weight << ")" << endl;
  }
@}

@d SII Edge Iterator reset
@{  cout << "------------ getting reset ---------" << endl;
  e_temp->reset();
  cout << "------------ getting five ---------" << endl;
  for (CORBA::ULong i = 0; i < e_itList->length(); ++i) {
        cout << "(" << (*e_itList)[i].node_i << "," <<
                (*e_itList)[i].node_j << "," <<
                (*e_itList)[i].weight << ")" << endl;
  }
@}

A few other calls to the graph interface round out the SII tests:

@d \verb|client| SII call second series
@{  cout << "Trying isConnected" << endl;
  if ( dg->isConnected(1) ) 
	cout << "Connected" << endl;
  if ( !( dg->isConnected(5) ) ) 
	cout << "Not Connected" << endl;

  nodeList->length(3);
  (*nodeList)[0] = 1;
  (*nodeList)[1] = 2;
  (*nodeList)[2] = 3;
  cout << "Scalar returned by addWeights: " 
	<< dg->addWeights( (*nodeList) ) << endl;

  stdGraphType * stgt = dg->getGraph();
  cout << "Trying getGraph ---------------" << endl;
  printEdges( &stgt->edgeList );
  printNodes( &stgt->nodeList );
@} 

Below is a concurrent call to a method returning a \verb|sequence|.
Notice that making this call requires more lines of code than do
static calls, placing a greater burden on the implementor.

@d \verb|client| DII call node list return
@{  cout << "Trying to set up a concurrent call" << endl;
  CORBA::Request_var _req = dg->_request("adjacentNodesOut");
  _req->add_in_arg( "node_name" ) <<= (CORBA::ULong)node_num; 
  _req->set_return_type(_tc_Nodes);
  _req->send_deferred();
  cout << "Sent Call, hopefully enough postage" << endl;
  _req->get_response();

  if( _req->env()->exception() ) {
    CORBA::Exception *_ex = _req->env()->exception();
    CORBA::UnknownUserException *_uuex = 
	CORBA::UnknownUserException::_downcast( _ex );
    if( _uuex )
      mico_throw( CORBA::UNKNOWN() );
    else mico_throw( *_ex );
  }

  Nodes* _res = new ::Nodes;
  *_req->result()->value() >>= *_res;
  nodeList = _res;

  cout << "Deferred Call Adjacent Nodes Out complete" << endl; 
  cout << "Printing out return: " << nodeList->length() << endl;
  for (CORBA::ULong i = 0; i < nodeList->length(); ++i) 
	cout << (*nodeList)[i] << endl;
@}

Below is an example of a DII call to an interface method with a scalar
return type.  Notice that much of the code involved in the call is
devoted to error-checking.

@d \verb|client| DII call scalar return
@{  cout << "Trying to set up a concurrent call" << endl;
  nodeList->length(3);
  (*nodeList)[0] = 1;
  (*nodeList)[1] = 2;
  (*nodeList)[2] = 3;
  _req = dg->_request("addWeights");
  _req->add_in_arg( "nodeList" ) <<= Nodes( (*nodeList) ); 
  _req->set_return_type(_tc_Weight);
  _req->send_deferred();

  cout << "Sent Call, hopefully enough postage" << endl;
  _req->get_response();

  if( _req->env()->exception() ) {
    CORBA::Exception *_ex = _req->env()->exception();
    CORBA::UnknownUserException *_uuex = 
	CORBA::UnknownUserException::_downcast( _ex );
    if( _uuex ) 
      mico_throw( CORBA::UNKNOWN() );
    else 
      mico_throw( *_ex );
  }

  Weight _res_weight;
  *_req->result()->value() >>= _res_weight;
  cout << "Scalar returned by addWeights: " 
	<< _res_weight << endl;

   cout << "CLEANING UP";
   dg->destroy();
   dg2->destroy();
   dg3->destroy();
   dg4->destroy();
   e_temp->destroy(); //destroy the iterator
   temp->destroy(); //destroy the iterator
@}



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\subsection{Client Using DII}

The below example client uses the CORBA Dynamic Invocation Interface 
to access server objects.

@o dclient_ns.cc -d 
@{@<\verb|client| includes@>
@<\verb|client| helper functions@>
int main ( int argc, char * argv[]) {
  @<\verb|client| orb setup@>
  @<\verb|client| nameservices@>
  @<\verb|dclient| setup code@>
  @<\verb|dclient| algorithm invocation@>
}
@}

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

@d \verb|dclient| setup code
@{  proxy_interface_var dg3 = fdg->createEmpty();

  cout << "Placing nodes ... " << endl;
  for( int i = 0; i < 9; ++i ) {
    cout << "Running with " << i << endl;
    dg3->addNode ( putNode( i ) );
  }
  cout << "Printing nodes ... " << endl;
  printNodes( dg3->getNodes() );

  cout << "Placing edges ... " << endl;
  dg3->addEdge ( putEdge(  0, 1, 4 ) );
  dg3->addEdge ( putEdge(  1, 2, 7 ) );
  dg3->addEdge ( putEdge(  2, 3, 7 ) );
  dg3->addEdge ( putEdge(  3, 4, 9 ) );
  dg3->addEdge ( putEdge(  4, 5, 10 ) );
  dg3->addEdge ( putEdge(  5, 6, 2 ) );
  dg3->addEdge ( putEdge(  6, 7, 1 ) );
  dg3->addEdge ( putEdge(  7, 0, 8 ) );
  dg3->addEdge ( putEdge(  1, 7, 11 ) );

  dg3->addEdge ( putEdge(  7, 8, 7 ) );
  dg3->addEdge ( putEdge(  2, 8, 2 ) );
  dg3->addEdge ( putEdge(  2, 5, 4 ) );
  dg3->addEdge ( putEdge(  3, 5, 14 ) );
  dg3->addEdge ( putEdge(  6, 8, 6 ) );

  proxy_interface_var dg = fdg->clone( dg3 );
  cout << "Allocated Graph" << endl;
@}

The below code demonstrates concurrent calls on the three main algorithms.

@d \verb|dclient| algorithm invocation
@{  cout << "Trying to set up multiple concurrent calls" << endl;
  vector <CORBA::Request_var> _req(3);

  @<\verb|dclient| request algorithms@>
  @<\verb|dclient| get back responses@>
  @<\verb|dclient| get back exceptions@>

  CORBA::Boolean cycled;
  *(_req[2])->result()->value() >>= 
	CORBA::Any::to_boolean ( cycled ) ;

  proxy_interface_var dg5;
  *(_req[0])->result()->value() >>=  dg5;  
  
  Nodes * node_list = new Nodes;
  *(_req[1])->result()->value() >>= *node_list;
  
  if ( cycled ) cout << "It does have cycles";
  else cout << "It does not have cycles";

  cout << "Printing nodes ... " << endl;
  printNodes( dg5->getNodes() );
  cout << "Printing edges ... " << endl;
  printEdges( dg5->getEdges() );

  cout << " Printing Topological Sort results " << endl;
  printNodes ( node_list );

  cout << "CLEANING UP" << endl;
  dg->destroy();
  dg3->destroy();
  dg5->destroy();
@}


@d \verb|dclient| request algorithms
@{  _req[0] = dg->_request("prims");
  _req[0]->add_in_arg( "r" ) <<= (CORBA::ULong)0; 
  _req[0]->set_return_type(_tc_proxy_interface);
  _req[0]->send_deferred();
  cout << "Sent Call, hopefully enough postage" << endl;

  _req[1] = dg->_request("topological_sort");
  _req[1]->add_in_arg( "u" ) <<= (CORBA::ULong)0; 
  _req[1]->set_return_type(_tc_Nodes);
  _req[1]->send_deferred();
  cout << "Sent Call, hopefully enough postage" << endl;

  _req[2] = dg3->_request("has_cycles");
  _req[2]->set_return_type(CORBA::_tc_boolean);
  _req[2]->send_deferred();
  cout << "Sent Call, hopefully enough postage" << endl;
@}

Below, the client gets the results of the algorithm calls. A sample 
exception-handling routine is included.

@d \verb|dclient| get back responses
@{  _req[0]->get_response();
  _req[1]->get_response();
  _req[2]->get_response();

@}

Below is an example of how to check the return code for exceptions
that might have occurred during this call.

@d \verb|dclient| get back exceptions
@{  // Sturdy Exception Code just for _req[0]
  if( _req[0]->env()->exception() ) {
    CORBA::Exception *_ex = _req[0]->env()->exception();
    CORBA::UnknownUserException *_uuex = 
        CORBA::UnknownUserException::_downcast( _ex );
    if( _uuex ) 
      mico_throw( CORBA::UNKNOWN() );
    else mico_throw( *_ex );
  }
@}
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\subsection{Timing-Test Client}

The below client implements a few timing tests comparing graph-service
calls in a non-distributed as well as in a distributed
setting.\footnote{The OSMisc::gettime calls are taken from the
examples/bench directory of the MICO 2.2.5 release.}

@o tclient_ns.cc -d
@{#include "matrixgraph.h"
#include "adjacencygraph.h"
@<\verb|client| includes@>
#include <mico/os-misc.h>
#include <iomanip.h>
@<\verb|client| helper functions@>
int main ( int argc, char * argv[]) {
  @<\verb|client| orb setup@>
  @<\verb|client| nameservices@>
  @<\verb|tclient| setup test code@>
  @<\verb|tclient| setup file@>
  @<\verb|tclient| addNodes individually@>
  @<\verb|tclient| addNodes all@>
  @<\verb|tclient| addEdges individually@>
  @<\verb|tclient| addEdges all@>
  @<\verb|tclient| cleanup@>
}
@}

@d \verb|tclient| setup test code
@{   matrix_graph<long, long> * mg1;
   adjacency_graph<long, long> * ag1;
 
   matrix_graph<long, long> * mg2;
   adjacency_graph<long, long> * ag2;
 

   proxy_interface_var dg1;
   proxy_interface_var dg2;
   int size, increment;
 
   fstream outFile;

   cout << "Enter the maximum size of the test: ";
   cin >> size;   
   cout << "Enter the increment for the test: ";
   cin >> increment;

   outFile.open("times.txt", ios::app);
   
@}

Below, the time taken for a series of calls to \verb|addNode| and
\verb|addEdge| is calculated in both a distributed and non-distributed
setting.

@d \verb|tclient| setup file
@{   outFile << "GT\t"
	   << "Size\t"
           << "LSN\t" << "DSN\t" << "LMN\t" << "DMN\t"
           << "LSE\t" << "DSE\t" << "LME\t" << "DME" 
	   << endl;

   CORBA::Long t1, t2;
   CORBA::Double lsn, dsn, lmn, dmn;
   CORBA::Double lse, dse, lme, dme;
   vector<int> stlnlist(size);
   vector<int>::iterator ni;
   Nodes corbanlist;
   corbanlist.length(size);
   vector<pair<int, int> > stlelist(size);
   vector<pair<int, int> >::iterator ei;
   Edges corbaelist;
   corbaelist.length(size);
@}
@d \verb|tclient| addNodes individually
@{
   for (int idx = 1; idx <= size; idx+=increment) {
     mg1 = new matrix_graph<long,long>;
     ag1 = new adjacency_graph<long,long>;
     mg2 = new matrix_graph<long,long>;
     ag2 = new adjacency_graph<long,long>;
     dg1 = fdg->createEmpty();
     dg2 = fdg->createEmpty();
     corbanlist.length(idx);
     corbaelist.length(idx);

     if ( input == 0 ) { 
	outFile << "adj\t";
     } else { 
	outFile << "mat\t";
     }

     outFile << idx << "\t";
     outFile << setiosflags(ios::fixed)
             << setprecision(3);

     if (input > 0) {
       t1 = OSMisc::gettime();
       for (int i = 0; i < idx; ++i) 
	 mg1->addNode(i);
       t2 = OSMisc::gettime();
     } else { 
       t1 = OSMisc::gettime();
       for (int i = 0; i < idx; ++i)
	 ag1->addNode(i);
       t2 = OSMisc::gettime();
     }
     lsn = (double)(t2-t1)/idx;

     t1 = OSMisc::gettime();
     for (int i = 0; i < idx; ++i) 
       dg1->addNode(i);
     t2 = OSMisc::gettime();
     dsn = (double)(t2-t1)/idx; 
@}

@d \verb|tclient| addNodes all
@{
    ni = stlnlist.begin();
    if (input > 0) {
       t1 = OSMisc::gettime();
       for (int i = 0; i < idx; ++i, ++ni) 
         stlnlist[i] = i;
       mg2->addNodes(stlnlist.begin(), ni);
       t2 = OSMisc::gettime();
     } else {
       t1 = OSMisc::gettime();
       for (int i = 0; i < idx; ++i, ++ni) 
         stlnlist[i] = i;
       ag2->addNodes(stlnlist.begin(), ni);
       t2 = OSMisc::gettime();
     }
     lmn = (double)(t2-t1)/idx;
   
     t1 = OSMisc::gettime();
     for (int i = 0; i < idx; ++i) 
       corbanlist[i] = i;
     dg2->addNodes( corbanlist );
     t2 = OSMisc::gettime();
     dmn = (double)(t2-t1)/idx;

@}


@d \verb|tclient| addEdges individually 
@{ 
     if (input > 0) {
       t1 = OSMisc::gettime();
       for (int i = 0; i < idx; ++i) { 
  	  mg1->addEdge( i, i+1, 3000);
	  mg1->addEdge( i+1, i, 3000);
       }
       t2 = OSMisc::gettime();
     } else {
       t1 = OSMisc::gettime();
       for (int i = 0; i < idx; ++i) {
  	  ag1->addEdge( i, i+1, 3000);
	  ag1->addEdge( i+1, i, 3000);
       }
       t2 = OSMisc::gettime();
     }
     lse = (double)(t2-t1)/idx; 

     t1 = OSMisc::gettime();
     for (int i = 0; i < idx; ++i) {
       dg1->addEdge( putEdge (i, i+1, 3000));
       dg1->addEdge( putEdge (i+1, i, 3000));
     }
     t2 = OSMisc::gettime();
     dse = (double)(t2-t1)/idx;
@}

@d \verb|tclient| addEdges all
@{
     ei = stlelist.begin();
     if (input > 0) {
       t1 = OSMisc::gettime(); 
       for (int i = 0; i < idx; ++i, ++ei ) { 
         stlelist[i].first = i; 
         stlelist[i].second = i + 1;
       }
       mg2->addEdges(stlelist.begin(), ei);
       t2 = OSMisc::gettime();
     } else {
       t1 = OSMisc::gettime();
       for (int i = 0; i < idx; ++i, ++ei ) { 
         stlelist[i].first = i; 
         stlelist[i].second = i + 1;
       }
       ag2->addEdges(stlelist.begin(), ei);
       t2 = OSMisc::gettime();
     }
     lme = (double)(t2-t1)/idx;  

     t1 = OSMisc::gettime();
     for (int i = 0; i < idx; ++i)
       corbaelist[i] = putEdge ( i, i+1, 3000 );
     dg2->addEdges( corbaelist );
     t2 = OSMisc::gettime();

     dme = (double)(t2-t1)/idx;
 
@}

This is where the transcription to the log is performed as well as the
release of memory allocated during this run:

@d \verb|tclient| cleanup
@{ 
     outFile << lsn << "\t" <<  dsn << "\t";
     outFile << lmn << "\t" <<  dmn << "\t";
     outFile << lse << "\t" <<  dse << "\t";
     outFile << lme << "\t" <<  dme << "\t" << endl;
     dg1->destroy();
     dg2->destroy();
     delete mg1;
     delete ag1;
     delete ag2;
     delete mg2;
   }

   outFile.close();
@}

\subsubsection{Timing Results} 

A comparison of times for calls on distributed graphs versus those for
their non-distributed counterparts has been obtained by running the
\verb|tclient|. We list the results of these tests by the appropriate
remote computer name.  The computers involved in these tests were
\verb|resnet-991|, \verb|cory|, and \verb|dishwasher|.

\begin{itemize} 

\item \verb|cory| is a Intel Pentium II machine clocked at 333Mhz
running Linux 2.2.4 with 64M of memory and 128 swap.

\item \verb|dishwasher| is a UltraSparc machine with dual 200Mhz CPUs
running Sun Solaris 5.6 with 400M of memory and 700 swap.

\item \verb|resnet-991| is a Cyrix 586 machine clocked at 166Mhz
running Linux 2.2.4 with 128M and 128 swap.  

\end{itemize}

\subsubsection{Testbed}
For an evaluation of performance across a wide range of graph
sizes, we have run these tests incrementally on graphs of varying sizes. 
Graphs with numbers of nodes 
incremented by 100 (up to a maximum of 2000) were tested. 
Running these tests on such a wide spectrum of sizes 
helps to smooth out data anomalies that may be caused by 
caching or other architectural ``sweet spots''\cite{Patterson}.

For each increase in size over the period of calculation, eight
distinct tests were performed.

\begin{itemize} \item Local Single Add Node (\verb|lsn|) invokes the
\verb|addNode| method to add nodes one at a time to a local graph.

\item Distributed Single Add Node (\verb|dsn|) invokes the
\verb|addNode| method to add nodes one at a time to a distributed
graph.

\item Local Multiple Add Node (\verb|lmn|) creates a \verb|vector| of
nodes and then calls the \verb|addNodes| method for a local graph.

\item Distributed Multiple Add Node (\verb|dmn|) creates a
\verb|sequence| of nodes and calls the \verb|addNodes| method for a
distributed graph.

\item Local Single Add Edge (\verb|lse|) invokes the \verb|addEdge|
method to add edges one at a time to a local graph.

\item Distributed Single Add Edge (\verb|dse|) invokes the
\verb|addEdge| method to add edges one at a time to a distributed
graph.
 
\item Local Multiple Add Edge (\verb|lme|) creates a \verb|vector| of
edges and then calls the \verb|addEdges| method for a local graph.

\item Distributed Multiple Add Edge (\verb|dme|) creates a
\verb|sequence| of edges and calls the \verb|addEdges| method for a
distributed graph.

\end{itemize}

Each group of tests was run with the three different machines acting
as servers.  The client always originated from \verb|resnet-991|.  The
structure of the files presented below is a \verb|ping| listing from
\verb|resnet-991| to the host, \verb|traceroute| from
\verb|resnet-991| to the host, and then the complete test for that
host generated by two runs of the client on \verb|resnet-991|.  The
client is run to generate a listing for the matrix and the adjacency
version.

\subsubsection{cory raw results}
@d cory-net
@{@i cory-net.txt
@}

@d resnet-cory 
@{@i resnet-cory.txt
@}

\subsubsection{dishwasher raw results}
@d dishwasher-net 
@{@i dishwasher-net.txt
@}

@d resnet-dishwasher
@{@i resnet-dishwasher.txt
@}

\subsubsection{resnet-991 raw results}
@d local-net
@{@i local-net.txt
@}

@d resnet-local
@{@i resnet-local.txt
@}

\subsubsection{Test Results}

The results of our timing tests reveal that adding nodes and edges
individually is much more expensive than adding sequences of nodes and
edges. Furthermore, they show that distributing computation to a
faster remote computer yields significant performance
benefits.\footnote{Incidentally, although no formal performance
metrics were performed, the distributed graph was able to allocate at
least 16000 nodes and edges in a sparse graph in which each node is
connected to the first node.}

When the tests are run entirely on a single local system, the
performance of that system degrades in a linear fashion as the CPU is
tasked to maintain both a client and a server. When the server is
moved off of the local system to \verb|cory|, advantages are gained
due to the faster processing speed of the remote machine---these
advantages become more striking with larger numbers of nodes and
edges.

When the server is moved to \verb|dishwasher|, the relatively small
network latency, along with the high speed of that machine's
processor, yields high performance in the \verb|DMN| and \verb|DME|
tests.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\section{A Simple Interactive Client}

The below interactive client demonstrates the sort of data-mapping
that a client will likely provide.

\subsection{Overview of the Source File}

@o iclient_ns.cc -d
@{
@<ic includes etc@>
@<\verb|client| helper functions@>

class nodeMapper {
  @<ic nodeMapper innards@>
};

int main ( int argc, char * argv[]) {
  @<ic data setup@>
  @<\verb|client| orb setup@>
  @<\verb|client| nameservices@>

  dg = fdg->create( (*graph) );
  cout << "Welcome. System is on line." << endl << endl;
  while( true ) {
    @<ic handle command line entries@>
  }  
  cout << endl << "Have a nice day!" << endl << endl;
  return 0;
} // end main()

@<ic \verb|grab_tokens| function@>
@<ic \verb|help| function|@>
@}


\subsection{Implementation}

@d ic includes etc
@{#include<iostream>
#include<string>
#include<vector>
#include<map>

@<\verb|client| includes@>

// forward refs to helper functions
vector<string> grab_tokens( bool& valid, int x );
void help();
@}

The below class provides for mapping of the integer data 
manipulated by our distributed graph system to data of another type 
(in this case, that type is \verb|string|).

\subsection{Mapping Nodes to Data}

@d ic nodeMapper innards
@{
private:
  typedef map<int, string> TableType;
  TableType table;
  int nextIndex;
public:
  nodeMapper() : nextIndex(0), table() {}
  ~nodeMapper() {}

  void insert_node( string x ) {
    table[nextIndex] = x;
    ++nextIndex;
  }
  void remove_node( int x ) {
    TableType::iterator i = table.find( x );
    if( i != table.end() )
    	table.erase( i );
  }
  int size() const { 
    return table.size();
  }
  string int_to_string( const int& x ) {
    TableType::iterator i = table.find( x );
    if( i != table.end() )
      return (*i).second;
    else return string("");
  }
  int string_to_int( const string& x ) const {
    TableType::const_iterator i;
    for( i = table.begin(); i != table.end(); ++i )
      if( (*i).second == x )
	return (*i).first;
  }
  bool name_exists( const string& x ) const {
    TableType::const_iterator i;
    for( i = table.begin(); i != table.end(); ++i )
      if( (*i).second == x )
	return true;
    return false;
  }
  bool int_exists( const int& x ) const {
    return( table.find( x ) != table.end() ); 
  }
@}

\subsection{Main Program}

@d ic data setup
@{  
  nodeMapper lookup;
  
  vector<string> lineIn;
  string tokenIn;
  bool valid;  // was it a valid command line?
  bool badcmd;
  vector<string> moreTokens;

  proxy_interface_var dg;
  stdGraphType* graph = new stdGraphType;

  Nodes* nodeList;
  Edges* edgeList;
@}

Requested services are performed on a single graph. All command lines 
must be terminated with the character `\verb|!|'.

@d ic handle command line entries
@{    @<ic reset prompt@>
    @<ic addedge request@>
    @<ic rmedge request@>
    @<ic addnode request@>
    @<ic rmnode request@>
    @<ic isconn request@>
    @<ic adjo request@>
    @<ic adji request@>
    @<ic ptnodes request@>
    @<ic ptedges request@>
    @<ic cycle request@>
    @<ic topsort request@>
    @<ic help request@>
    @<ic quit request@>
    @<ic handle bad command line@>
@}


@d ic reset prompt
@{    moreTokens.erase( moreTokens.begin(), moreTokens.end() );
    badcmd = false;
    cout << "bkp> ";
    cin >> tokenIn;
@}

\subsubsection{Making Proxy Calls}

It is in the below code that the calls to the distributed graph proxy 
are made. Some checking of entered data is performed locally 
before these calls are made.

@d ic addedge request
@{    if( tokenIn == "addedge" ) { 
      moreTokens = grab_tokens(valid, 2);
      if( lookup.name_exists( moreTokens[0] )
	  && lookup.name_exists( moreTokens[1] ) ) {        
        dg->addEdge( putEdge( 
              lookup.string_to_int( moreTokens[0] ),
              lookup.string_to_int( moreTokens[1] ),
              (CORBA::ULong)(0) ));
      }
      else cout << "Cannot add edge to/from"
                << " nonexistent node!" << endl;
    }
@}

@d ic rmedge request
@{    else if( tokenIn == "rmedge" ) {
      moreTokens = grab_tokens(valid, 2);
      if( lookup.name_exists( moreTokens[0] ) 
	  && lookup.name_exists( moreTokens[1] ) ) {
        dg->removeEdge( lookup.string_to_int( moreTokens[0] ),
                        lookup.string_to_int( moreTokens[1] ) );
      }
      else cout << "That edge does not exist!" << endl;
    }
@}

@d ic addnode request
@{    else if( tokenIn == "addnode" ) {
      moreTokens = grab_tokens(valid, 1);
      if( !lookup.name_exists( moreTokens[0] ) ) {
        lookup.insert_node( moreTokens[0] );
        dg->addNode( lookup.string_to_int( moreTokens[0] ) );
      }
      else cout << "That node already exists!" << endl;
    }
@}

@d ic rmnode request
@{    else if( tokenIn == "rmnode" ) {
      moreTokens = grab_tokens(valid, 1);
      if( lookup.name_exists( moreTokens[0] ) ) {
        Nodes tmpEdges(1);
        tmpEdges[0] = lookup.string_to_int( moreTokens[0] );
        lookup.remove_node( 
          lookup.string_to_int( moreTokens[0] ) );
        dg->removeNodes( tmpEdges );
      }
      else cout << "Node doesn't exist!!" << endl;
    }
@}

@d ic isconn request
@{    else if( tokenIn == "isconn" ) {
      moreTokens = grab_tokens(valid, 1);
      if( lookup.name_exists( moreTokens[0] ) ) {
         if( dg->isConnected( 
               lookup.string_to_int( moreTokens[0] ) ) )
           cout << "Yes, that node is connected.";
         else
           cout << "No, that node is not connected.";
         cout << endl;
      }
      else cout << "Node doesn't exist!!" << endl;
    }
@}

@d ic adjo request
@{    else if( tokenIn == "adjo" ) {
      moreTokens = grab_tokens(valid, 1);
      if( lookup.name_exists( moreTokens[0] ) ) {
        nodeList = dg->adjacentNodesOut( lookup.string_to_int( 
                               moreTokens[0] ) );
        for( int i = 0; i < (*nodeList).length(); ++i )
	  cout << lookup.int_to_string( (*nodeList)[i] ) << endl;
      }
      else cout << "Node doesn't exist!!" << endl;
    }
@}

@d ic adji request
@{    else if( tokenIn == "adji" ) {
      moreTokens = grab_tokens(valid, 1);
      if( lookup.name_exists( moreTokens[0] ) ) {
        nodeList = dg->adjacentNodesIn( lookup.string_to_int( 
                               moreTokens[0] ) );
        for( int i = 0; i < (*nodeList).length(); ++i )
	  cout << lookup.int_to_string( (*nodeList)[i] ) << endl;
      }
      else cout << "Node doesn't exist!!" << endl;
    }
@}

@d ic ptnodes request
@{    else if( tokenIn == "ptnodes" ) {
      grab_tokens(valid, 0);
      nodeList = dg->getNodes();
      for( int i = 0; i < (*nodeList).length(); ++i ) {
        string node1( lookup.int_to_string( (*nodeList)[i] ));
        if( lookup.name_exists( node1 ) )
	  cout << lookup.int_to_string( (*nodeList)[i] ) << endl;
      }
      delete nodeList;
    }
@}

@d ic ptedges request
@{    else if( tokenIn == "ptedges" ) {
      grab_tokens(valid, 0);
      edgeList = dg->getEdges();
      for( int i = 0; i < (*edgeList).length(); ++i ) {
        string node1 = lookup.int_to_string( 
                              (*edgeList)[i].node_i );
        string node2 = lookup.int_to_string( 
                              (*edgeList)[i].node_j );
        if( lookup.name_exists( node1 ) 
          && lookup.name_exists( node2 ) ) 
            cout << node1 << " --> " << node2 << endl;
      }
      delete edgeList;
    }
@}

@d ic cycle request
@{    else if( tokenIn == "cycle?" ) {
      grab_tokens(valid, 0);
      if( dg->has_cycles() )
	cout << "The graph has a cycle.";
      else cout << "The graph has no cycles.";
      cout << endl;
    }
@}

@d ic topsort request
@{    else if( tokenIn == "topsort?" ) {
      moreTokens = grab_tokens(valid, 1);
      if( lookup.name_exists( moreTokens[0] ) ) {
	nodeList = dg->topological_sort(
              lookup.string_to_int( moreTokens[0] ) );
        for( int i = 0; i < (*nodeList).length(); ++i )
          cout << lookup.int_to_string( (*nodeList)[i] ) << endl;
	delete nodeList;
      }
      else cout << "Node doesn't exist!!" << endl;
    }
@}

@d ic help request
@{    else if( tokenIn == "help" ) {
      grab_tokens(valid, 0);
      help();
      valid = true; // allow other toks
    }
@}

@d ic quit request
@{    else if( tokenIn == "quit" ) {
      grab_tokens(valid, 0); // all done
      break;
    }
@}

@d ic handle bad command line
@{    else {
      grab_tokens(valid, 0);
      badcmd = true;
      cout << endl << "!!error!! Command " 
	   << tokenIn << " not recognized!" << endl << endl;
    }
    if( !valid && !badcmd )
      cout << endl << "!!error!! Bad input line!" << endl << endl;
@}

Finally, the below utility methods are part of this client program:

\subsection{Utilities}

@d ic \verb|grab_tokens| function
@{vector<string> grab_tokens( bool& valid, const int x ) {
  vector<string> tmpTokens;
  string curToken("");
  while( curToken != "!" ) {
    cin >> curToken;
    if( curToken != "!" )
      tmpTokens.push_back( curToken );
  }
  if( tmpTokens.size() == x )
    valid = true;
  else valid = false;
  
  return tmpTokens;
}
@}

@d ic \verb|help| function|
@{void help() {
  cout << endl << endl;
  
  cout << "Valid commands are: " << endl;
  cout << "    addedge # #" << endl;
  cout << "    rmedge # # " << endl;
  cout << "    addnode #" << endl;
  cout << "    rmnode #" << endl;
  cout << "    isconn #" << endl;
  cout << "    adjo #" << endl;
  cout << "    adji #" << endl;
  cout << "    ptnodes" << endl;
  cout << "    ptedges" << endl;
  cout << "    cycle?" << endl;
  cout << "    topsort? #" << endl;
  cout << "    help" << endl;
  cout << "    quit" << endl;
  cout << endl;
}
@}

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\section{Startup Script}

@o graph_proxy 
@{@<\verb|graph_proxy| graph proxy script@>
@<\verb|graph_proxy| graph proxy ns@>
@<\verb|graph_proxy| graph proxy imr@>
@<\verb|graph_proxy| graph proxy client@>
@}
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

This script is meant to be run on UNIX systems with the Bourne
shell. To invoke this script after running \verb|nuweb| on
\verb|graph_proxy.w| you will need to type ``\verb|sh graph_proxy|''.
Four flags are supported: \verb|-f| will run the first client
presented in this paper, \verb|-d| will cause the DII client to run,
\verb|-t| invokes the timing-test client, and \verb|-i| runs the
interactive client.

This script starts and kills the naming service after the client has
been run once.  The servers are run in \verb|shared| mode; see
pp. 34-35 of \cite{Roemer} for a discussion of the various server
activation styles.

@d \verb|graph_proxy| graph proxy script
@{
#!/bin/sh

PATH=../../daemon:../../coss/naming:../../imr:../../ir:@$PATH
ADDR=inet:`uname -n`:12456
RC="-ORBImplRepoAddr @$ADDR -ORBNamingAddr @$ADDR"

echo Starting up Mico daemon
micod -ORBIIOPAddr @$ADDR &
micod_pid=@$!
sleep 1

trap "kill @$micod_pid" 0
@}

The naming service must be ``started up.'' 

@d \verb|graph_proxy| graph proxy ns
@{
echo "registering name service ..."
imr create NameService poa `which nsd` \
   IDL:omg.org/CosNaming/NamingContext:1.0#NameService @$RC

@}

Next, the servers are registered and then activated.

@d \verb|graph_proxy| graph proxy imr
@{# register server
echo Starting up imr registration NS stuff
imr create mgraph_proxy_factory shared "`pwd`/mserver_ns @$RC" \
    IDL:graph_proxy:1.0 @$RC \

# register server
echo Starting up imr registration NS stuff
imr create agraph_proxy_factory shared "`pwd`/aserver_ns @$RC" \
    IDL:graph_proxy:1.0 @$RC \

echo Registering with NSD
imr activate mgraph_proxy_factory @$RC
imr activate agraph_proxy_factory @$RC

sleep 1
@}

The \verb|sleep| command allows enough time for the servers to be
activated before the client attempts to contact them.

@d \verb|graph_proxy| graph proxy client
@{
case @$1 in
  "-s")
    echo Starting server on @$ADDR
    echo Contact with -ORBNamingAddr @$ADDR
    sleep 9000
    ;;
  "-d")
    echo Starting: ./dclient_ns -ORBNamingAddr @$ADDR
    ./dclient_ns -ORBNamingAddr @$ADDR
    ;;
  "-i")
    echo Starting: ./iclient_ns -ORBNamingAddr @$ADDR
    ./iclient_ns -ORBNamingAddr @$ADDR
    ;;
  "-t")
    echo Starting: ./tclient_ns -ORBNamingAddr @$ADDR
    ./tclient_ns -ORBNamingAddr @$ADDR
    ;;
  "-f")
    echo Starting: ./client_ns -ORBNamingAddr @$ADDR
    ./client_ns -ORBNamingAddr @$ADDR
    ;;
  *)
    echo Must supply a client flag: -f, -d, -i or -t
esac
@}

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\section{Makefile}
@o Makefile -t
@{
@<\verb|Makefile| Makefile@>
@}

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

The following UNIX makefile provides for compilation of the programs
presented in this paper, as well as for production of this formatted
documentation.

@d \verb|Makefile| Makefile
@{
  @<\verb|Makefile| vars@>

all:  servers clients

servers: a@$(SERVER) m@${SERVER}
clients: @${CLIENT} d@$(CLIENT) t@$(CLIENT) i@$(CLIENT)

@<\verb|Makefile| server compilation@>
@<\verb|Makefile| client compilation@>
@<\verb|Makefile| ndisgraph compilation@>
@<\verb|Makefile| document production@>

@$(PRJ).o: @$(PRJ).cc
	mico-c++ @$(CPPFLAGS) -I. -c @$(PRJ).cc -o @$(PRJ).o

@$(PRJ).h @$(PRJ).cc: @$(PRJ).idl
	idl @$(IDLFLAGS) @$(PRJ).idl

@$(PRJ).idl: @$(PRJ).w
	nuweb @$(PRJ).w
	chmod o+x @$(PRJ)

#code-gen:
#       idl --use-dii --c++-impl @${IDLNAME).idl

clean:
	rm -fr *.o ?@${SERVER} ?@${CLIENT} @${CLIENT} *~ \
		*.tex *.log core *.idl *.inf *.toc\
		*.dvi *.ps *.aux *.cc *.h @${PRJ} \
		mdemo ademo
@}

Note that although \cite{Roemer}, p. 15, states that use of the flag
\verb|--use-dii| has no affect on application implementation. However,
in practice, we have found this not to be the case. If one intends to
make DII calls from a client, the IDL must be run with
\verb|--use-dii|.

@d \verb|Makefile| vars
@{
	# For the distributed examples
	VERSION = 2.2.5
	PRJ = graph_proxy
	SERVER = server_ns
	CLIENT = client_ns
	IDLFLAGS = --use-dii
	CPPFLAGS = -g
	MICLIB = -lmico@$(VERSION) -lmicocoss@$(VERSION)

	# For the non distributed graph examples
	BASEFILES=graph.h graph_interface.h
	MATRIXFILES=matrixgraphbase.h matrixgraph.h \
	main.cc @$(BASEFILES)
	ADJFILES=adjacencygraphbase.h adjacencygraph.h\
	 main2.cc @$(BASEFILES)
	FLAGS=-ftemplate-depth-30
@}

@d \verb|Makefile| server compilation
@{
m@$(SERVER): @$(PRJ).o 
	mico-c++ -DMAT @$(CPPFLAGS) -I. \
-c @$(SERVER).cc -o m@$(SERVER).o
	mico-ld -I. @$(PRJ).o m@$(SERVER).o @$(MICLIB) -o m@$(SERVER)

a@$(SERVER): @$(PRJ).o 
	mico-c++ -DADJ @$(CPPFLAGS) -I. \
-c @$(SERVER).cc -o a@$(SERVER).o
	mico-ld -I. @$(PRJ).o a@$(SERVER).o @$(MICLIB) -o a@$(SERVER)
@}

@d \verb|Makefile| client compilation
@{
@$(CLIENT): @$(PRJ).o 
	mico-c++ @$(CPPFLAGS) -I. -c @$(CLIENT).cc -o @$(CLIENT).o
	mico-ld -I. @$(PRJ).o @$(CLIENT).o @$(MICLIB) -o @$(CLIENT)

d@$(CLIENT): @$(PRJ).o 
	mico-c++ @$(CPPFLAGS) -I. -c d@$(CLIENT).cc -o d@$(CLIENT).o
	mico-ld -I. @$(PRJ).o d@$(CLIENT).o @$(MICLIB) -o d@$(CLIENT)

i@$(CLIENT): @$(PRJ).o 
	mico-c++ @$(CPPFLAGS) -I. -c i@$(CLIENT).cc -o i@$(CLIENT).o
	mico-ld -I. @$(PRJ).o i@$(CLIENT).o @$(MICLIB) -o i@$(CLIENT)

t@$(CLIENT): @$(PRJ).o 
	mico-c++ @$(CPPFLAGS) -I. -c t@$(CLIENT).cc -o t@$(CLIENT).o
	mico-ld -I. @$(PRJ).o t@$(CLIENT).o @$(MICLIB) -o t@$(CLIENT)
@}

The targets \verb|mdemo| and \verb|ademo| prompt compilation 
of the don-distributed graph tests.

@d\verb|Makefile| ndisgraph compilation
@{
mdemo:  @$(MATRIXFILES)
	mico-c++ main.cc @$(FLAGS) -o mdemo

ademo:  @$(ADJFILES)
	mico-c++ main2.cc @$(FLAGS) -o ademo
@}

The target \verb|doc| produces this postscript document.

@d \verb|Makefile| document production
@{
doc: @$(PRJ).w
	nuweb @$(PRJ).w; latex @$(PRJ).tex
	nuweb @$(PRJ).w; latex @$(PRJ).tex
	dvips @$(PRJ).dvi -o @$(PRJ).ps
@}

\newpage
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\section{Limitations}

Currently, CORBA has several limitations that hinder creation of a
purely distributed environment. Many ORB vendors' proprietary ORB
features hinder portability from one ORB to another. Furthermore, ORB
interoperability is far from guaranteed.

Furthermore, we found production of a Java client to be impractical in
the scope of this project due to the fact that use of the ORB provided
by Sun Microsystems as part of Java SDK 2 requires storage and
transmission of lengthy IORs for interoperability with the \verb|MICO|
ORB. Kay Roemer, co-creator of \verb|MICO| has said, ``...the JDK uses
some proprietary mechanism to find initial references via those
ORBInitial properties. The only portable way to make the JDK ORB use
MICO's naming service is to pass the stringified IOR of mico's naming
service to the Java application and turn it into an
Object...''\cite{Roemer:mail}

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\section{Conclusion}

In implementing our distributed graph system, we have found the use of
proxy components to be an advantageous development approach. Proxy
components enable high modularization and permit existing
non-distributed systems to be migrated to a distributed setting
without making changes to the source code of those existing systems.

Furthermore, we have found that a distributed scheme allows a user of
a computation-intensive system like our graph system to get the
results of computations more quickly. A server may be placed on a
remote machine that is faster, or which has been otherwise optimized
for the sort of computation to be performed.  Even taking into account
the overhead of network communication, significant time savings can
then be realized, as shown by the results of our timing-test client
runs.

\newpage
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\section{Appendix A: Non-Distributed Graphs}

In this appendix, we present two implementations of a generic STL
graph container used in the system described in this paper. Our
primary goal in the design of these containers was to facilitate end
users' creation of graph classes with varying internal data
structures, such as matrices and lists. For the purposes of
demonstration, we have fully implemented a matrix graph and will
discuss how to create other representations easily using this
framework.  Then, we present an adjacency-list graph representation
which was built upon this framework.

The class hierarchy, in order from base class to the final graph
class, is as follows:
 \verb|Graph|, \verb|GraphInterface|, followed by the
implementation-specific classes: \verb|MatrixGraphBase| and
\verb|matrix_graph| and the instantiable class.  The
implementation-specific files for our adjacency-list representation
will be presented thereafter. Both implementations have been designed
for compliance with the STL \verb|Container| concept.

Other, established graph packages 
\cite{Knuth}\cite{GTL} exist. Proxies similar to those developed in 
this document may be developed for such packages.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\subsection{Graph Interface Reference}

Listed here are all of the operations supported by both graph
implementations.  The label \verb|GraphType| will, in the below
listing, serves to represent, interchangeably, either a matrix graph
or adjacency graph.


\begin{itemize}

\item \verb|GraphType()|---Constructor; the graph is assumed to be undirected.

\item \verb|GraphType(bool isDigraph)|---Constructor, 
specifying via a boolean flag indicating whether the graph is directed.

\item \verb|GraphType( const GraphType& g )|---Copy constructor.

\item \verb|virtual void addNode ( const DataType& nodeName )|---Adds
a node of the specified name.

\item \verb|void addNodes( ForwardIterator i1, ForwardIterator i2 )|---\newline
Adds a collection of nodes to the graph. The forward iterators
passed in as parameters correspond to the locations in a container
bounding a sequence of elements to be added to the graph.

\item \verb|Scalar addWeights ( ForwardIterator i1, ForwardIterator i2 )|\newline
\verb|const|---
Sums the weights of the edges between a series of nodes
of a specific order. The two parameters should point into a container of
node names. If two or more consecutive nodes do not have an edge between
them, no value is added to the total for the edge sequence.

\item \verb|void adjacentNodesIn ( const DataType& nodeName,|\newline
\verb|OutputIterator o)|---
Outputs, via the output iterator, all edges spanning into \verb|nodeName|.

\item \verb|void adjacentNodesOut ( const DataType& nodeName,|\newline
\verb|OutputIterator o)|---
Outputs, via the output iterator, all edges spanning out of \verb|nodeName|.

\item \verb|iterator begin()|---Returns an iterator to the first node
in the node container.

\item \verb|const_iterator begin() const|---Returns a constant iterator to
the first node in the node container.

\item \verb|void clear()|---Removes all nodes and edges.

\item \verb|bool empty() const|---Returns \verb|true| if the container is
empty (no nodes exist). Otherwise, \verb|false| is returned.

\item \verb|iterator end()|---Returns an iterator to the end element in
the node container.

\item \verb|const_iterator end()|---Returns a constant iterator to the
end element in the node container.

\item \verb|virtual Scalar findWeight ( const DataType& nodeName1, const|\newline
\verb|DataType& nodeName2 ) const|---
Finds the weight of the edge between the two nodes given. 
If an edge does not exist between the two nodes, scalar zero is returned.

\item \verb|void getEdges ( OutputIterator o ) const|---Places all edges
(node-name pairs) into an arbitrary container pointed to by the
output iterator.

\item \verb|virtual bool isConnected(const DataType& nodeName) const|\newline
---Returns \verb|true| if the specified node is not an orphan. Otherwise,
it returns \verb|false|.

\item \verb|void listAll( ostream& o ) const|---Fancy-prints all node
pairs in the container to the output stream.

\item \verb|Scalar max_size() const|---
Returns the maximum number of nodes in the node container.

\item \verb|virtual Scalar numEdges() const|---Returns the number of
edges in the graph.

\item \verb|reverse_iterator rbegin()|---Returns a reverse iterator
pointing to the last node in the node container.

\item \verb|const_reverse_iterator rbegin() const|---Returns a constant
reverse iterator pointing to the last node in the node container.

\item \verb|reverse_iterator rend()|---Returns a reverse iterator
pointing to the first node in the node container.

\item \verb|const_reverse_iterator rend() const|---Returns a constant
reverse iterator pointing to the first node in the node container.

\item \verb|virtual void removeNode(const DataType& nodeName)|---\newline
Removes the node with the specified name.

\item \verb|void removeNodes(ForwardIterator i1, ForwardIterator i2)|
\newline--- Removes a collection of nodes. The forward iterators
passed in as parameters correspond to locations bounding a sequence of
node names which are to be deleted from this container.

\item \verb|void removeAllNodes()|---Removes all nodes (and therefore
all edges) from the graph. Has the exact same functionality as \verb|clear()|.

\item \verb|virtual void removeOrphanNodes()|---This method removes
all orphan nodes from the graph.

\item \verb|Scalar size() const|---Returns the size of (number of nodes in)
the container.

\item \verb|virtual void swap(GraphType& g)|---Swaps the contents
of two graph objects.

\item \verb|GraphType& operator=() const|---The assignment operator.

\end{itemize}

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\subsection{The Graph Class}

The \verb|Graph| class, the root class in the hierarchy, provides operations
that \emph{any} graph will need, regardless of implementation.

\verb|Graph| maintains a unique mapping between node ``names'' and an
internal scalar key.  The node ``names'' are assigned by the end
user--- in the demonstration programs, we have chosen the name of be
of type \verb|string|, however, any type that supports the
\verb|operator==| operation should work in this context.  The internal
scalar key associates a scalar value with a any given node and is
useful to some subclasses (such as the matrix) for indexing purposes.

It would probably be more efficient to dispose of the container which
stores these pairings and simply handle any necessary mappings in the
subclasses. However, this would put a greater burden on the designer
of these implementation classes.  The focus in the design of the base
classes is more on modularity and less on maximizing efficiency. The
idea is to make it easy for programmers to create new graph
implementations using this framework; storing key/name pairs at the
\verb|Graph| level helps realize this goal.

The contents of \verb|graph.h| are as follows:

@o graph.h -d
@{#ifndef GRAPH_H
#define GRAPH_H
@<\verb|Graph| directives, header files, and declarations@>
@<The \verb|one_less_than| adaptor definition@>
@<The \verb|equal2nd| adaptor definition@>
@<\verb|Graph| class definition and protected data members@>
@<\verb|Graph| constructors@>
@<Other methods for the \verb|Graph| class@>
@<\verb|Graph| code for adding and removing nodes@>
@<Information extraction methods for the \verb|Graph| class@>
@<Protected methods for use by subclasses of \verb|Graph|@>
@<External overloaded output operations for \verb|Graph|@>
@<\verb|graph_iterator| class definition@>
#endif
@}

%%%%%%%%%%%%%%%%%%%%%%  GRAPH MACROS HERE %%%%%%%%%%%%%%%%%%%%%%%%%%


We begin by using a compiler directive to prevent recompilation of the
header and include the necessary header files; we include such a
directive in each of our source files.  To avoid having to specify
namespaces explicitly, we use the \verb|using| construct to
automatically assume the ``std'' namespace.

@d \verb|Graph| directives, header files, and declarations
@{#include <map>
#include <string>
#include <algorithm>
#include <pair.h>
#include <function.h>
#include <iostream.h>
#include <fstream>
#include <typeinfo>
#include "pairhelp.h"

using namespace std;

#define MAX_NODENAME_LENGTH     80
@}

The below adaptor, a \verb|BinaryPredicate|, will be used with the STL
\verb|adjacent|\newline\verb|_find| algorithm. Its purpose is to
assist in finding ``gaps'' in sequences of scalar indices.  If nodes
are added to a graph, these nodes automatically will be assigned
sequentially increasing index values, starting with the first non-zero
scalar value (``1`` in the case of integers).  If an element other
than that at the end of the sequence is removed later on, it will, of
course, no longer exist in the node-container.

By applying \verb|adjacent_find| using the below adaptor, we can
identify the first non-consecutive (non-existent) scalar value in the
node container.  The idea is to insert new nodes at the first ``gap``
to keep minimal the size of the graph data representation.  (It is
assumed that the container used for node pairings must exhibit an
ordering of elements based on key (scalar) values).

Below, one can see that \verb|operator()| is defined to check that the
\verb|first| element (the key) in a pair contains a value one less
than the next in the ordered container.  It will return \verb|true| as
long as this is the case and \verb|false| otherwise, causing the
\verb|adjacent_find| algorithm to halt on the first non-consecutive
element.

@d The \verb|one_less_than| adaptor definition
@{template <class Pair>
class one_less_than
{
public:
  typedef Pair first_argument_type;
  typedef Pair second_argument_type;
  typedef bool result_type;

  bool operator() (const Pair& p1, const Pair& p2) const {
    return (p2.first == p1.first+1); 
  }
};
@}

The next adaptor is a \verb|Predicate|. This adaptor simply allows for
comparison of the value of a ``name'' element to the \verb|second|
element in a scalar/name pair.

@d The \verb|equal2nd| adaptor definition
@{template <class Pair, class CompareType>
class equal2nd 
{
  private:
    CompareType equal_val;

  public:
    typedef Pair argument_type;

    equal2nd(CompareType t) : equal_val(t) {}
    bool operator() (const Pair& p) { return p.second==equal_val; }
};
@}


Now we come to the definition of the \verb|Graph| class.  Upon
instantiation, the user must specify, via template parameters, the
data types that will be used as ``key'' (must be a scalar type) and
``name''.

The protected section of the class contains a few type definitions to
simplify naming. We revealed an egcs 1.0.3 compiler bug when
attempting to compile one of the type definitions. To work around this
problem, we have added support for changing this typedef to a macro
definition if the \verb|EGCS_TEMPLATE_BUG| macro is defined during
compilation.  If this macro is used, text substitutions are made by
the preprocessor rather than during compilation itself.

After the type-definitions are two object declarations.  One is for
the ordered container (described previously) that stores scalar/data
pairs.  The second is a boolean flag indicating whether or not the
given graph is directed.

@d \verb|Graph| class definition and protected data members
@{template<class KeyType, class DataType>
class Graph {

protected:
  typedef KeyType  Scalar;
  typedef pair<KeyType,DataType> GraphPair;  
 
  #ifndef EGCS_TYPEDEF_BUG
  typedef map<KeyType,DataType,less<KeyType> > NodeContainer;  
  #else
  #define NodeContainer map<KeyType,DataType,less<KeyType> >
  #endif

   NodeContainer nodeContainer;
   bool isDigraph;
@}


The \verb|Graph| class supplies three constructors. 
The third is a copy constructor.

@d \verb|Graph| constructors
@{public:
  explicit Graph () : isDigraph(false)  { }

  explicit Graph (bool isDi) : isDigraph(isDi)  { }

  Graph ( const Graph& g ) : nodeContainer(g.nodeContainer), 
	                     isDigraph(g.isDigraph) {}
@}

The \verb|addNode()| method adds a node name to the container and
automatically assigns it an internal scalar index, starting with 1.
The scalar value and node name are kept in an STL \verb|pair|.  The
\verb|adjacent_find| algorithm, in conjunction with the
\verb|one_less_than| adaptor (described above), finds gaps in index
numbers within the container and assigns a new node to a ``gap''
index, if one exists. Subclasses must override this method by first
calling this version of the method and then performing any
implementation-specific bookkeeping that may be necessary.

The \verb|removeNode| method removes an existing node-pair from the
container. Like \verb|addNode|, subclasses must override this method by first
calling this version and then performing any implementation-specific tasks.

@d \verb|Graph| code for adding and removing nodes
@{  virtual void addNode( const DataType& nodeName )  {
    NodeContainer::iterator i;

    if (nodeContainer.empty())
      nodeContainer[1]=nodeName;

    else if((i=adjacent_find(nodeContainer.begin(), 
      nodeContainer.end(), 
      not2( one_less_than<GraphPair>()))) != 
            nodeContainer.end())
      nodeContainer[(*i).first+1] = nodeName;
    else nodeContainer[nodeContainer.size()+1] = nodeName;
  }

  virtual void removeNode ( const DataType& nodeName )  { 

    NodeContainer::iterator i = 
        find_if ( nodeContainer.begin(), 
                  nodeContainer.end(), 
                  equal2nd<GraphPair,DataType>(nodeName) );         
   if (i != nodeContainer.end())
    nodeContainer.erase(i);
  }

  virtual void removeAllNodes()  {  nodeContainer.clear(); }
@}


The \verb|Graph| class provides only a few methods for
node-information extraction--- the \verb|size|, \verb|getIndexOf|, and
\verb|listAll| methods, which are all constant. \verb|size| returns
the number of nodes in the graph; \verb|getIndexOf| returns the index
of a given node-data item (this information has proved indispensable
in algorithm implementation).  \verb|listAll| fancy-prints all node
pairings to an \verb|ostream|.

@d Information extraction methods for the \verb|Graph| class
@{  
  virtual Scalar size() const { return nodeContainer.size(); }

  virtual Scalar getIndexOf ( const DataType& nodeName ) const {
    NodeContainer::const_iterator i =
      find_if( nodeContainer.begin(), nodeContainer.end(),
               pair_second_equal_to<DataType>(nodeName));
      
    Scalar scale(0);
    while (i-- != nodeContainer.begin()) 
          scale++;
    return scale;	     
  }

  virtual void listAll(ostream& o) const {
    NodeContainer::const_iterator first=nodeContainer.begin(), 
                                  last=nodeContainer.end();

    o << string("Scalar/NodeName Node Pairs for this ") + 
         (isDigraph ? "directed" : "undirected") << " graph:" 
         << endl;

    while (first != last)
    {   o << "(" << (*first).first << ", " << 
             (*first).second << ")" << endl; first++; 
    }
  }
@}


The below operations are required by the STL \verb|Container|concept:

@d Other methods for the \verb|Graph| class
@{  Graph& operator=( const Graph& g ) {
    if ( &g == this )
      return *this;

    nodeContainer = g.nodeContainer;
    isDigraph = g.isDigraph;           
    return *this;
  }

  virtual void swap ( Graph& g ) {
    nodeContainer.swap( g.nodeContainer );
    bool tmp = isDigraph;
    isDigraph = g.isDigraph;	 
    g.isDigraph = tmp;
  }
@}


The following methods are supplied for use by subclasses.

The \verb|nodeExists()| method determines whether a node of a
specified ``name'' exists in the graph and returns the truth value of
its existence as a \verb|bool|.

The \verb|getScalarOf()| and \verb|getIndexOf()| methods are
similar but have one subtle difference:  The former returns
the actual key value of the pair in the container for
a given node. The latter returns an entry from the consecutive numbering
of nodes in the container. For example, if the third
element in the container had a key value of 7, \verb|getScalarOf()|
would return 7, whereas \verb|getIndexOf()| would return 2 (meaning
the third ordered element, assuming 0 is a valid index).

The \verb|nodeNameByIndex()| retrieves the ``name'' of a node
based on an index number. A precondition to this method is that
the index is valid.

@d Protected methods for use by subclasses of \verb|Graph|
@{protected:
  bool nodeExists ( const DataType& nodeName ) const {
    return (find_if( nodeContainer.begin(), nodeContainer.end(),
      pair_second_equal_to<DataType>(nodeName)) != 
                              nodeContainer.end());
  }
            
  // Precondition- nodeName is a *valid* node name.
  Scalar getScalarOf ( const DataType& nodeName ) const {
    return (*(find_if( nodeContainer.begin(), nodeContainer.end(), 
             pair_second_equal_to<DataType>(nodeName)))).first;
  }

  DataType nodeNameByIndex ( Scalar index ) const {
    NodeContainer::const_iterator i = nodeContainer.begin();
    Scalar x(0);
    while ( x < index ) { x++; i++; }
    return (*i).second;
  }
};
@}

The following output operation calls the \verb|listAll()| method
described above for any \verb|ostream| object.  It is declared
globally because the compiler must distinguish between types
on a global scope when determining which version of \verb|operator<<|
to execute during scope resolution.

@d External overloaded output operations for \verb|Graph|
@{template <class KeyType, class DataType>
ostream& operator<< (ostream& o, const Graph<KeyType,DataType>& g)
{
  g.listAll(o);
  return o;
}

@}

The \verb|graph_iterator|, defined below, is provided
for use via the type-definitions \verb|iterator|, \verb|reverse_iterator|,
\verb|const_|\verb|iterator| and \verb|const_reverse_iterator|. The
following iterator class is actually an adaptor class for all
of the iterators supported by the node container.  It also
models the \verb|ForwardIterator| concept.  The entire class
definition is as follows:

@d \verb|graph_iterator| class definition
@{template <class Iterator, class DataType>
class graph_iterator
{
private:
  Iterator i;

public:
  typedef bidirectional_iterator_tag  iterator_category;
  typedef ptrdiff_t                   difference_type;
  typedef DataType                    value_type;
  typedef value_type&                 reference;
  typedef value_type*                 pointer;
  typedef graph_iterator    	      self;
  typedef self&                       self_ref;

  graph_iterator() {}

  graph_iterator( Iterator j ) : i(j) {}

  self  operator++(int) {
    self tmp = *this;
    ++(*this);
    return tmp;
  }

  self& operator++() { i++; return (*this); }
  self& operator--() { i--; return (*this); }
  self  operator--(int) {
    self tmp = *this;
    *this--;
    return tmp;
  }
  
  bool operator==(const graph_iterator& it) const
  { return (i== it.i); }

  bool operator!=(const graph_iterator& it) const
  { return (i != it.i); }

  reference operator*()
  { return (*i).second;  }

  pointer operator->()
  { return &(*i).second; }
};
@}

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\subsection{The GraphInterface Class}

The \verb|GraphInterface| class defines all methods that need to be
supported by any implementation.  Most methods in this class are
therefore abstract, making it necessary for subclasses to implement
them for successful compilation. End users of graph implementations
will find the available primitive graph operations (services) here.

The contents of the \verb|graph_interface.h| file are as follows:

@o graph_interface.h -d
@{#ifndef GRAPH_INTERFACE_H
#define GRAPH_INTERFACE_H
@<\verb|GraphInterface| directives and header files@>
@<Class definition and constructors for \verb|GraphInterface|@>
@<\verb|GraphInterface| public interface@>
#endif
@}


%%%%%%%%%%%%%%%%%%  GRAPHINTERFACE MACROS HERE %%%%%%%%%%%%%%%%%%%%%

Here are the directives and header files for the \verb|GraphInterface|
class:

@d \verb|GraphInterface| directives and header files
@{#include "graph.h"
#include <vector>
@}


The \verb|GraphInterface| class supports constructors
analogous to those defined in the \verb|Graph| class. No additional
is done at this level---the parent constructors are simply invoked from here.

@d Class definition and constructors for \verb|GraphInterface|
@{template <class KeyType,class DataType>
class GraphInterface : public Graph<KeyType,DataType>
{
public:
  typedef KeyType Scalar;

  explicit GraphInterface() : Graph() {}

  explicit GraphInterface(bool isDi) : Graph(isDi) {}

  GraphInterface( const GraphInterface& g ) : Graph(g) {}
@}

These are the new public interface methods, in addition to
those few defined in \verb|Graph|. Again, most are abstract; those
that are not are those that cannot be due to the use of templates.

The \verb|AdjacentNodesIn()| and \verb|AdjacentNodesOut()| methods
both return (via an \verb|OutputIterator|) a sequence of nodes which have
edges leading into or out of, respectively, the specified node.

The \verb|isConnected()| method determines whether a given
node is connected by an edge to any other nodes in the graph.

\verb|findWeight()| returns the weight of an edge between
two specified nodes.

\verb|addWeights()| sums the weights between nodes in an
ordered list. A precondition to this method is that edges
exist between the nodes.

\verb|removeOrphanNodes()| removes nodes that are not connected
to any other node in the graph.

\verb|removeNodes()| is written in terms of \verb|removeNode()| and
allows the user to eliminate several nodes in one method call
using iterators that point into an arbitrary container of node names.

\verb|addEdge()| and \verb|removeEdge()|, as the names imply,
add and remove edges from the graph. These methods are purely
implementation-specific, as there is no notion of an ``edge''
in the \verb|Graph| class.

\verb|addEdges()| adds a series of edges to a graph.  The
forward iterators specified as arguments should point
into a container of node-name pairs.

\verb|getEdges()| outputs all edges in the graph as node-name pairs
into using an output iterator.

@d \verb|GraphInterface| public interface
@{  template <class OutputIterator>
  bool adjacentNodesOut(const DataType& nodeName, 
                              OutputIterator o) const;

  template <class OutputIterator>
  bool adjacentNodesIn(const DataType& nodeName, 
                              OutputIterator o) const;

  virtual bool isConnected(const DataType& nodeName) const=0;

  virtual Scalar findWeight ( const DataType& nodeName1, 
                              const DataType& nodeName2 ) const=0;


  template < class ForwardIterator >
  Scalar addWeights ( ForwardIterator iterator1, 
                      ForwardIterator iterator2 ) const;

  virtual void removeOrphanNodes()=0;
 
  template <class ForwardIterator>
  void removeNodes ( ForwardIterator iterator1, 
                     ForwardIterator iterator2 );

  virtual bool addEdge(const DataType& nodeName1, 
                       const DataType& nodeName2,
                       const Scalar weight)=0;

  template <class ForwardIterator>
  bool addEdges( ForwardIterator begin, ForwardIterator end );

  virtual bool removeEdge(const DataType& nodeName1,
                          const DataType nodeName2)=0;

  template <class OutputIterator>
  void getEdges ( OutputIterator o );   
};
@}

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\subsection{The MatrixGraphBase Class}

Up until this point, we have given source and documentation
for the underlying framework of any graph implementation.
However, a user wishing to create a class implementing a graph in a
particular way will derive that implementation from \verb|GraphInterface|
and provide the required functionality.

This section presents an implementation of one particular type of
graph class---the matrix-based graph. We will briefly discuss issues
pertaining to the successful derivation of other user-defined graph
classes from those already given, with the intention of making
creation other implementations rather simple.

Other types of graph classes need not conform to the programming 
style of this implementation,
as long as all required methods are implemented
(correctly).

The \verb|matrixgraphbase.h| file, containing the matrix-based
graph implementation and supporting classes, contains the
following:

@o matrixgraphbase.h -d
@{#ifndef MATRIX_GRAPH_BASE_H
#define MATRIX_GRAPH_BASE_H
@<\verb|MatrixGraphBase| directives and header files@>
@<The \verb|Edge| class definition@>
@<\verb|MatrixGraphBase| class definition and private data members@>
@<\verb|MatrixGraphBase| constructors and destructors@>
@<Container methods for \verb|MatrixGraphBase|@>
@<\verb|MatrixGraphBase| interface methods@>
@<Protected methods for \verb|MatrixGraphBase|@>
@<External overloaded output operations for \verb|MatrixGraphBase|@>
#endif
@}

%%%%%%%%%%%%%%%%  MATRIXGRAPHBASE MACROS HERE %%%%%%%%%%%%%%%%%%%%%
Here are the directives and header-file inclusion for \verb|matrixgraphbase.h|:

@d \verb|MatrixGraphBase| directives and header files
@{#include "graph_interface.h"
@}



The following class represents an edge. In a matrix-based
implementation, every matrix entry consists of an \verb|Edge| object.
The private attributes of the \verb|Edge| object consist of the
``edge'' flag, specifying whether or not the edge exists, as well as a
``hidden'' flag, which allows an existing edge to be ignored.  There
also exists a ``weight'' attribute holding a generic scalar value and
represents the weight of the given edge.

Most of the methods for this class are straightforward. However,
we would like to point out a subtle difference between the methods
\verb|edgeExists()| and \verb|isEdge()|.  The former method
specifies whether an edge exists whether it is hidden or not---that
is, it ignores the hidden attribute of the edge.  The latter
specifies that the edge exists only if it is both an edge and is
not hidden.

@d The \verb|Edge| class definition
@{template< class Scalar >
class Edge
{
private:
  bool edge, hidden;
  Scalar weight;

public:
  Edge() : hidden(false), edge(false), weight(0) { }

  void makeEdge( Scalar scale ) { edge=true; weight=scale; }

  void removeEdge() { edge=false; }

  bool isEdge() const { return (edge && !hidden); }

  bool edgeExists() const { return edge; }

  Scalar getWeight() const {  return weight; }

  void setWeight(Scalar wgt) { weight=wgt; }

  bool isHidden() const { return hidden; }

  bool hide() { if (isEdge()) 
                 return false; 
                else hidden=true;
                return true;    }

  bool unhide() { if (!isHidden()) 
                   return false;
                  else
                    hidden=false;
                  return true;    }
};
@}

Now we present the class definition, type-definitions and private data
members for the \verb|MatrixGraphBase| class.  The template parameters
are the same as those for \verb|GraphInterface|.  The
\verb|MatrixRows| container is a two-dimensional vector, where a
vector of \verb|MatrixColumns| are the elements of each
\verb|MatrixRows| index.  STL vectors allow the indexing scheme used
to access matrix elements to be the same as if it were done using
primitive C++ multidimensional arrays.

@d \verb|MatrixGraphBase| class definition and private data members
@{template<class KeyType,class DataType>
class MatrixGraphBase : 
  public GraphInterface<KeyType,DataType> {  
  private:       
    typedef KeyType Scalar;
    typedef Edge<Scalar> EdgeType;
    typedef vector<EdgeType> MatrixColumns;
    typedef vector<MatrixColumns> MatrixRows;

    MatrixRows       rows;
    NodeContainer    tempContainer;
@}

These are the constructors for the \verb|MatrixGraphBase| class.
Again, they have the same signatures and semantics as the 
constructors in the \verb|GraphInterface| and \verb|Graph| classes.

@d \verb|MatrixGraphBase| constructors and destructors
@{public:
  explicit MatrixGraphBase() : 
           GraphInterface() {}


  explicit MatrixGraphBase(bool isDi) : 
           GraphInterface(isDi) {}

  MatrixGraphBase ( const MatrixGraphBase& g ) : 
        GraphInterface(g), rows(b.rows), tempContainer() {}

  ~MatrixGraphBase() {}
@}

The below protected methods are for internal use by this class and its
subclasses.  In this particular application, these methods aren't used
by subclasses, but are available in case the developer decides he
needs them.  The allocation methods, \verb|allocateGraph()|,
\verb|deAllocateGraph()| and \verb|reallocateGraph| are used for
initializing the matrix to the appropriate size
(\verb|allocateGraph()|), deleting the matrix
(\verb|deAllocateGraph()|) or restructuring an existing matrix
(\verb|reAllocateGraph()|.  The \verb|reAllocateGraph()| method is
called whenever a node is deleted from the graph.  Because the node
has been deleted, any prior references to the node must be eliminated
and the appropriate rows and columns updated within the matrix.

@d Protected methods for \verb|MatrixGraphBase|
@{  void allocateGraph() {
  if (nodeContainer.size()>0)
    for (Scalar x(0); x < nodeContainer.size();x++)
      rows.push_back( MatrixColumns(nodeContainer.size()));       
  }

  void deAllocateGraph() {
    rows.clear();
  }

  void reAllocateGraph() {
    if (!rows.empty())
    {
      NodeContainer::iterator it;
      pair<NodeContainer::iterator, NodeContainer::iterator>
        matchPair =
           mismatch ( tempContainer.begin(), 
                      tempContainer.end(), 
                      nodeContainer.begin());
      // Nodes were added
      while (tempContainer.size() < nodeContainer.size()) {
        Scalar count(0);
        it = matchPair.second; 
        while (it-- != nodeContainer.begin()) count++;

        tempContainer.insert( *matchPair.second );
        rows.insert ( rows.begin()+count, 
                      MatrixColumns(tempContainer.size()));
        for (Scalar y(0); y < rows.size(); ++y)
          rows[y].insert(rows[y].begin()+count, EdgeType());
      }
      // Nodes were removed
      while (tempContainer.size() > nodeContainer.size()) {
        Scalar count(0);
        it = matchPair.first;
                  
        while (it-- != tempContainer.begin()) count++;

        tempContainer.erase(matchPair.first);
        rows.erase( rows.begin()+count );   
        for (Scalar y(0); y < rows.size(); y++)
          rows[y].erase( rows[y].begin()+count );   
      }
    }    
    tempContainer.clear();
  }
};
@}

The following methods implement the abstract methods in the 
\verb|GraphInterface| class. For a brief explanation
of the purpose of each method, please see the documentation for the
\verb|GraphInterface| class above.

@d \verb|MatrixGraphBase| interface methods
@{
@<Matrix implementation of \verb|adjacentNodesOut|@>
@<Matrix implementation of \verb|adjacentNodesIn|@>
@<Matrix implementation of \verb|isConnected|@>
@<Matrix implementation of \verb|findWeight|@>
@<Matrix implementation of \verb|addWeights|@>
@<Matrix implementation of \verb|removeOrphanNodes|@>
@<Matrix implementation of \verb|addNode|@>
@<Matrix implementation of \verb|addNodes|@>
@<Matrix implementation of \verb|removeNode|@>
@<Matrix implementation of \verb|removeNodes|@>
@<Matrix implementation of \verb|removeAllNodes|@>     
@<Matrix implementation of \verb|addEdge|@> 
@<Matrix implementation of \verb|addEdges|@> 
@<Matrix implementation of \verb|removeEdge|@> 
@<Matrix implementation of \verb|getEdges|@> 
@<Matrix implementation of \verb|numEdges|@> 
@<Matrix implementation of \verb|listAll|@> 
@}

@d Matrix implementation of \verb|adjacentNodesOut|
@{  template  <class OutputIterator>
  bool adjacentNodesOut(const DataType& nodeName, 
                        OutputIterator o) const {
    if( numEdges() == (Scalar)(0) )
      return false;
    if (!nodeExists(nodeName))
      return false;
    Scalar index = getIndexOf(nodeName);
    const MatrixColumns& columns = rows[index];
    NodeContainer::const_iterator nodeIterator=
      nodeContainer.begin();
    MatrixColumns::const_iterator matrixIterator=columns.begin();

    for (Scalar x(0); matrixIterator != columns.end(); 
                      matrixIterator++, nodeIterator++, x++)
      if ( (*matrixIterator).isEdge() && x != index )
        *o = (*nodeIterator).second;
     
    return true;
  }
@}


@d Matrix implementation of \verb|adjacentNodesIn|
@{  template <class OutputIterator>
  bool adjacentNodesIn(const DataType& nodeName, 
                             OutputIterator o) const {
    if (!nodeExists(nodeName))
      return false;
    Scalar index = getIndexOf(nodeName);
    NodeContainer::const_iterator nodeIterator=
      nodeContainer.begin();

    for (Scalar x(0); x < rows.size(); x++, nodeIterator++) {
      if (x==index) // Skip the row corresponding to "nodeName"
        continue;

      if (rows[x][index].isEdge())
        *o = (*nodeIterator).second;
    }    
    return true;
  }
@}

@d Matrix implementation of \verb|isConnected|
@{  virtual bool isConnected(const DataType& nodeName) const {
    if( numEdges() == (Scalar)(0) )
      return false;
    if (!nodeExists(nodeName))
      return false;
    Scalar index = getIndexOf(nodeName);
    const MatrixColumns& columns = rows[index];

    for ( Scalar x(0); x < columns.size(); x++ )
      if (columns[x].isEdge() && x != index) return true;

    for ( Scalar x(0); x < rows.size(); x++)
      if (rows[x][index].isEdge() && x != index) return true;

    return false;
  }
@}

@d Matrix implementation of \verb|findWeight|
@{  virtual Scalar findWeight ( const DataType& nodeName1, 
                              const DataType& nodeName2 ) const {
    if (!nodeExists(nodeName1) | !nodeExists(nodeName2))
      return Scalar(0);

    return rows[getIndexOf(nodeName1)]
               [getIndexOf(nodeName2)].getWeight();
  }
@}

@d Matrix implementation of \verb|addWeights|
@{ template < class ForwardIterator >
  Scalar addWeights ( ForwardIterator iterator1,
                      ForwardIterator iterator2 ) const {
    Scalar scale(0);
    ForwardIterator tmp=iterator1;

    while (++iterator1 != iterator2)
      scale = scale + findWeight( *tmp++, *iterator1 );

    return scale;
  }
@}

@d Matrix implementation of \verb|removeOrphanNodes|
@{  virtual void removeOrphanNodes()  {
    vector<DataType> removeContainer;

    for ( NodeContainer::iterator i=nodeContainer.begin(); 
                                 i != nodeContainer.end(); i++ )
    {
      if (!isConnected( (*i).second ))
        removeContainer.push_back( (*i).second );
    }
    removeNodes ( removeContainer.begin(), removeContainer.end() );
  }
@}

@d Matrix implementation of \verb|addNode|
@{  virtual void addNode ( const DataType& nodeName ) {
    copy ( nodeContainer.begin(), nodeContainer.end(),
           inserter(tempContainer, tempContainer.begin()));

    Graph::addNode(nodeName);
    reAllocateGraph();  
  }
@}

@d Matrix implementation of \verb|addNodes|
@{  template <class ForwardIterator>
  void addNodes ( ForwardIterator iterator1, 
                  ForwardIterator iterator2 ) {
    while (iterator1 != iterator2)
    { addNode(*iterator1); iterator1++; }
  }
@}

@d Matrix implementation of \verb|removeNode|
@{  virtual void removeNode ( const DataType& nodeName ) {
    copy ( nodeContainer.begin(), nodeContainer.end(), 
    inserter(tempContainer, tempContainer.begin()));
 
    Graph::removeNode(nodeName);
    reAllocateGraph();
  }
@}

@d Matrix implementation of \verb|removeNodes|
@{  template <class ForwardIterator>
    void removeNodes ( ForwardIterator iterator1, 
                       ForwardIterator iterator2 )
    {
         while ( iterator1 != iterator2 ) {
            removeNode ( *iterator1 );
            iterator1++;
	  }
     }
@}

@d Matrix implementation of \verb|removeAllNodes|
@{  virtual void removeAllNodes() {
    Graph::removeAllNodes();
    deAllocateGraph();
  }
@}

@d Matrix implementation of \verb|addEdge|
@{  virtual bool addEdge(const DataType& nodeName1, 
                       const DataType& nodeName2,
                       const Scalar weight=0)
  {
    if (!nodeExists(nodeName1) || !nodeExists(nodeName2))
      return false;

    if (rows.empty())
      allocateGraph();

    Scalar index1 = getIndexOf(nodeName1);
    Scalar index2 = getIndexOf(nodeName2);

    rows[index1][index2].makeEdge(weight);

    if (!isDigraph && nodeName1 != nodeName2 )
      rows[index2][index1].makeEdge(weight);

    return true;
  }
@}

@d Matrix implementation of \verb|addEdges|
@{  template <class ForwardIterator>
  bool addEdges ( ForwardIterator begin, ForwardIterator end ) {
    for (; begin != end;  begin++) 
      if (!addEdge( (*begin).first, (*begin).second ))
        return false;
    return true;       
  }
@}

@d Matrix implementation of \verb|removeEdge|
@{  virtual bool removeEdge(const DataType& nodeName1, 
                          const DataType nodeName2)
  {
    if (!nodeExists(nodeName1) || !nodeExists(nodeName2))
      return false;

    Scalar index1 = getIndexOf(nodeName1);
    Scalar index2 = getIndexOf(nodeName2);

    rows[index1][index2].removeEdge();

    if (!isDigraph && nodeName1 != nodeName2 )
      rows[index2][index1].removeEdge();

    return true;
  }
@}

@d Matrix implementation of \verb|getEdges|
@{  template <class OutputIterator> 
  void getEdges ( OutputIterator o ) const {
    for (Scalar x(0); x < rows.size(); x++)
      for (Scalar y(0); y < rows[x].size(); y++)
      {
        if (x == y) continue;
        if (rows[x][y].isEdge())
        {  
          const pair<DataType,DataType> p(nodeNameByIndex(x),
                                        nodeNameByIndex(y));
          *o = p;
        }
     }     
  }
@}

@d Matrix implementation of \verb|numEdges|
@{  virtual Scalar numEdges() const {
    Scalar count=0;

    for (Scalar x(0); x < rows.size();x++)
      for (Scalar y(0); y < rows[x].size(); y++)
        if (rows[x][y].isEdge() && x != y )
          count++;

    if (!isDigraph)
      count /= 2;
    return count;
  }
@}

@d Matrix implementation of \verb|listAll|
@{  virtual void listAll(ostream& o) const {
    Graph::listAll(o);

    o << endl << "Edge listings: "; 
    if (rows.size()) o << endl;
    else o << "None." << endl;

    MatrixColumns::const_iterator mci;
    MatrixRows::const_iterator mri;
    NodeContainer::const_iterator nciX, nciY;

    for (mri=rows.begin(), nciX=nodeContainer.begin(); 
                           mri != rows.end(); mri++, nciX++)
      for (mci= (*mri).begin(), nciY=nodeContainer.begin(); 
                                mci != (*mri).end(); mci++, nciY++)
      {
        if ( (*mci).isEdge())
          cout << (*nciX).second << " has an edge to " << 
                  (*nciY).second << " of weight: " << 
                  (*mci).getWeight() << endl; 
      }
  }
@}

The following methods are required for compliance with the STL
\verb|Container| concept. They perform the necessary work
specific to matrix-graphs and also call the \verb|Graph| version
of each.

@d Container methods for \verb|MatrixGraphBase|
@{public:
  MatrixGraphBase& operator= ( const MatrixGraphBase& g ) {
    if ( &g == this )
      return *this;
  
    Graph::operator=(g);

    rows = g.rows;
    tempContainer.clear();
    return *this;
  }

  virtual void swap ( MatrixGraphBase& g ) {
    Graph::swap(g);
    rows.swap(g.rows);
    tempContainer.swap(g.tempContainer);
  }
@}


Here is an overloaded output operator specific to the \verb|MatrixGraphBase|
class:

@d External overloaded output operations for \verb|MatrixGraphBase|
@{template<class KeyType, class DataType>
ostream& operator<< (ostream& o, 
                     const MatrixGraphBase<KeyType,DataType>& g)
{
  g.listAll(o);
  return o;
}
@}


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\subsection{The Instantiable Matrix-Based Graph}

The \verb|matrix_graph| class resides in the file \verb|matrixgraph.h|
and is the lowest (instantiable) class in the hierarchy.  This class
conforms to the STL \verb|Forward-|\newline\verb|Container| and
\verb|ReversibleContainer| concepts.  Among these requirements is an
iterator for traversing the nodes within the container; and all other
methods required by \verb|Container| are also given.

An outline of the \verb|matrixgraph.h| file is as follows:

@o matrixgraph.h -d
@{#ifndef MATRIX_GRAPH_H
#define MATRIX_GRAPH_H
@<\verb|matrix_graph| directives and header files@>
@<\verb|matrix_graph| class definition and type-definitions@>
@<Constructors for \verb|matrix_graph|@>
@<\verb|matrix_graph| container wrappers for STL compliance@>
#endif
@}


%%%%%%%%%%%%%%%%%%  MATRIXGRAPH MACROS HERE %%%%%%%%%%%%%%%%%%%%%%%

Once again, the directives and header file:

@d \verb|matrix_graph| directives and header files
@{#include "matrixgraphbase.h"
@}

The \verb|matrix_graph| class, defined below,
is the final product to be manipulated by the end user and is suitable
for use with any appropriate STL algorithm.  The template parameters
are the same as those for \verb|MatrixGraphBase| and
\verb|GraphInterface|.  The type-definitions required by an STL
\verb|ForwardContainer| and \verb|ReversibleContainer| are also given:

@d \verb|matrix_graph| class definition and type-definitions
@{template <class KeyType, class DataType>
class matrix_graph : 
  public MatrixGraphBase<KeyType,DataType>
{
public:
  typedef DataType                        value_type;
  typedef DataType&                       reference_type;
  typedef const DataType&                 const_reference;
  typedef DataType*                       pointer;
  typedef ptrdiff_t                       difference_type;
  typedef size_t                          size_type;
  typedef graph_iterator
    <NodeContainer::iterator,DataType> iterator;
  typedef graph_iterator
    <NodeContainer::const_iterator,DataType> const_iterator;
  typedef graph_iterator
    <NodeContainer::reverse_iterator,DataType>
       reverse_iterator;
  typedef graph_iterator
    <NodeContainer::const_reverse_iterator,DataType>
      const_reverse_iterator;
@}


Listed below are the constructors, which follow the same semantics
as the constructors for the previous classes.

@d Constructors for \verb|matrix_graph|
@{  explicit matrix_graph( bool isDirected ) :
                         MatrixGraphBase(isDirected) {}

  explicit matrix_graph () : MatrixGraphBase() {}

  matrix_graph ( const matrix_graph& g ) :
                       MatrixGraphBase(g) { }
@}


Here are the methods required for STL compliance.  They are, more or less,
wrappers for the node container's methods, since it is the container
that the user will actually be accessing. Other methods merely
call the original method from a base class.

@d \verb|matrix_graph| container wrappers for STL compliance
@{  iterator begin() { return nodeContainer.begin(); }
  const_iterator begin() const { return nodeContainer.begin(); }
  iterator end() { return nodeContainer.end(); }
  const_iterator end() const { return nodeContainer.end(); }
  reverse_iterator rbegin() { return nodeContainer.rbegin(); }
  const_reverse_iterator rbegin() const 
    { return nodeContainer.rbegin(); }
  reverse_iterator rend() { return nodeContainer.rend(); }
  const_reverse_iterator rend() const 
    { return nodeContainer.rend(); }
  bool empty() const { return nodeContainer.empty(); }
  void clear() { removeAllNodes(); deAllocateGraph(); return; }  
  void swap ( matrix_graph& g ) { GraphInterface::swap(g); }
  Scalar max_size() const { return nodeContainer.max_size(); }

  matrix_graph& operator=( const matrix_graph& g ) {
    MatrixGraphBase::operator=(g);
    return *this;
  }
};
@}

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\subsection{The pairhelp.h Header File}

The \verb|pairhelp.h| header file contains methods for
testing equality between an arbitrary element and the first or second
elements of a pair.  They do not contain any code
specific to graphs and therefore may be reused in other
applications that may require access to a pair element for
purposes of testing equality.

@o pairhelp.h -d
@{#ifndef PAIRHELP_H
#define PAIRHELP_H
@<\verb|pair_first_equal_to| class definition@>
@<\verb|pair_second_equal_to| class definition@>
#endif
@}


%%%%%%%%%%%%%%%%%%  PAIRHELP_H MACROS HERE %%%%%%%%%%%%%%%%%%%%%%%

Class \verb|pair_first_equal_to| is a \verb|Predicate| which
accepts an arbitrary type via its constructor that must be the
same as the type of the \verb|first| pair element.  Later, when the \verb|operator()|
method is called, a \verb|pair| is passed in as an argument
and compared to the element passed in during construction.
The operator returns \verb|true| if the elements are equal
and \verb|false| otherwise.

@d \verb|pair_first_equal_to| class definition
@{template <class ConstructorType>
class pair_first_equal_to
{
private:
  const ConstructorType& ct;
public:
  pair_first_equal_to ( const ConstructorType& t ) : ct(t) {}

  template <class Type>
  bool operator() ( const Type& t ) { return (t.first == ct); }
};
@}



The second \verb|Predicate| is the same as the first, except
that the \verb|second| pair element is the one checked for
equality against the item passed in during construction:

@d \verb|pair_second_equal_to| class definition
@{template <class ConstructorType>
class pair_second_equal_to
{
private:
  const ConstructorType& ct;

public:
  pair_second_equal_to ( const ConstructorType& t  ) : ct(t) {}

  template <class Type>
  bool operator() ( const Type& t ) { return (t.second == ct); }
 };
@}

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\subsection{Matrix Graph Test Application}

The below test application demonstrates some of the capabilities of
the \verb|matrix_| \verb|graph| class.

@o main.cc -d
@{@<Header files for the main program@>
@<Declare a \verb|matrix_graph| object@>
@<First test@>
@<Second test@>
@<Third test@>
@<Fourth test@>
@<Fifth test@>
@}


%%%%%%%%%%%%%%%%%%%%  MAIN.CC MACROS HERE %%%%%%%%%%%%%%%%%%%%%%%%


First, we include the header file for the \verb|matrix_graph| class:

@d Header files for the main program
@{#include "matrixgraph.h"
@}

Then, we declare the graph object we will use.

@d Declare a \verb|matrix_graph| object
@{int main()
{
  matrix_graph<unsigned int,string> g(true);
@}

We start by adding four nodes and demonstrating how to iterate through our graph container.
After that, we add a couple of edges, demonstrate how
output the contents (nodes) of the graph, and finally show how
large the container is using \verb|size()|.

@d First test
@{  g.addNode("New York");
  g.addNode("Fort Lauderdale");
  g.addNode("Los Angeles");
  g.addNode("Ridgefield");

  cout << "A listing of all nodes using the graph iterator: ";
  copy ( g.begin(), g.end(), 
         ostream_iterator<string>(cout, "\n"));
  cout << endl << endl;

  g.addEdge("New York", "Los Angeles", 3000);
  g.addEdge("Fort Lauderdale", "New York", 1500);

  cout << g << endl;
  cout << "There are " << g.size()
       << " nodes in the graph" << endl;
@}

Next, we remove a node and add a different node and a new edge.
This demonstrates how a ``gap'' in the node sequence is left by the removal of
the Fort Lauderdale node and also to show that any edges leading
to or from Fort Lauderdale were removed.

@d Second test
@{  g.removeNode("Fort Lauderdale");
  g.addNode("Nowheresville");
  g.addEdge("New York", "Nowheresville", 50);
  cout << g;
@}

Next, we test the \verb|adjacentNodesOut()| and
\verb|adjacentNodesIn()|. Note that an arbitrary container can
be used to store the results of these two calls---for this
demonstration, we arbitrarily chose a \verb|vector|.

After demonstrating those two functions, we proceed to show
that Ridgefield is not connected to any other node by using
the \verb|isConnected()| method, and finally adding an
edge to Ridgefield from Los Angeles.

@d Third test
@{  vector<string> adjacent;

  g.adjacentNodesOut("New York", back_inserter(adjacent));

  cout << "List of nodes to which edges lead out of"
       << " New York: " << endl;
  copy (adjacent.begin(),adjacent.end(),
        ostream_iterator<string>(cout, "   "));
  cout << endl << endl;

  adjacent.clear();
  g.adjacentNodesIn("Los Angeles", back_inserter(adjacent));

  cout << "List of nodes from which edges lead into"
       << " Los Angeles: " << endl;
  copy (adjacent.begin(),adjacent.end(),
        ostream_iterator<string>(cout, "   "));
  cout << endl << endl;

  if ( g.isConnected("Ridgefield") )
    cout << "Ridgefield is connected." << endl;
  else 
    cout << "Ridgefield is not connected." << endl;

   g.addEdge("Los Angeles", "Ridgefield", 3030);

  if ( g.isConnected("Ridgefield") )
    cout << "Ridgefield is connected." << endl;
  else 
    cout << "Ridgefield is not connected." << endl;
@}

Next, we demonstrate the \verb|addWeights()| function
by populating a container with a list of nodes and passing an 
iterator to this container to the method.
The method accumulates the total weight between
all three cities and this is subsequently output to the screen.

@d Fourth test
@{  vector<string> cities(3);
  cities[0] = "New York";
  cities[1] = "Los Angeles";
  cities[2] = "Ridgefield";

  cout << "The distance between NY, LA and Ridgefield is: " << 
    g.addWeights( cities.begin(), cities.end() ) << endl;
@}

We next demonstrate the removal of orphan nodes.  To do this,
we create an unconnected node named New City;  we see that it is
removed after the call to \verb|removeOrphanNodes()|.

@d Fifth test
@{  g.addNode("New City");

  cout << "The graph before removing orphans: "
       << endl << g << endl;
  g.removeOrphanNodes();
  cout << "The graph after removing orphans: " << endl << g;
 
 return 0;
}
@}

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% ADJACENCY LIST %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\subsection{The AdjacencyGraphBase Class}

We have also implemented another type of graph class---the
adjacency-list-based graph.  This is another example of the sort of
graph class that may be created that conforms to
\verb|GraphInterface|. It has also, like \verb|matrix_graph|, been
designed for compliance with the STL \verb|Container| concept.

@o adjacencygraphbase.h
@{#ifndef ADJACENCY_GRAPH_BASE_H
#define ADJACENCY_GRAPH_BASE_H
@<\verb|AdjacencyGraphBase| directives and header files@>
@<This implementation's \verb|Edge| class@>
@<\verb|AdjacencyGraphBase| class definition and private data members@>
@<\verb|AdjacencyGraphBase| constructors and destructors@>
@<Container methods for \verb|AdjacencyGraphBase|@>
@<\verb|AdjacencyGraphBase| interface methods@>
@<Protected methods for \verb|AdjacencyGraphBase|@>
@<External overloaded output operations for \verb|AdjacencyGraphBase|@>
#endif
@}


This implementation, as required, derives from \verb|GraphInterface|,
the source for which must be included below.
In creating an adjacency list, we will make use of the STL \verb|list| class.

@d \verb|AdjacencyGraphBase| directives and header files
@{#include "graph_interface.h"
#include<list>
@}


As in \verb|MatrixGraph|, we define an Edge class that will represent
an edge joining two nodes in our graph.
Since this is an adjacency-list representation,
such a representation must include an identifier for the pointed-to node.
The index of the pointed-to node cannot be inferred from a position in the data structure,
as is the case with a matrix representation.

This additional information is stored in the private member \verb|itsIndex|,
which can be accessed via the public method \verb|index|.
Additional methods, \verb|incIndex| and \verb|decIndex|,
are provided for decrementing and incrementing indices.
Aside from creation via a copy constructor, these two methods are the only ways \verb|itsIndex|
can be modified. The reason they are needed in this graph implementation will,
hopefully, become apparent when we examine \verb|adjacency_graph|'s \verb|removeNode| and \verb|addNode| methods.

@d This implementation's \verb|Edge| class
@{template< class Scalar >
class aEdge {
private:
  bool hidden;
  Scalar weight, itsIndex;
public:
  aEdge( Scalar index ) :
    hidden(false), weight(0), itsIndex( index ) {}
  aEdge( Scalar index, Scalar scale ) :
    hidden(false), weight(scale), itsIndex( index ) {}
  Scalar index() const { return itsIndex; }
  Scalar getWeight() const {  return weight; }
  void setWeight(Scalar wgt) { weight=wgt; }
  void decIndex() { if( itsIndex > 0 ) --itsIndex; }
  void incIndex() { ++itsIndex; }
  bool isHidden() const { return hidden; }
  bool hide() { 
    if (isHidden()) return false;
    else hidden=true;
    return true;
  }

  bool unhide() {
    if (!isHidden()) return false;
    else hidden=false;
    return true;
  }
};
@}


From the below typedefs, we see that this graph representation stores edges in
a vector of lists. An instance, \verb|adjList|, of this type (\verb|AdjType|)
is allocated and remains private to this class.

@d \verb|AdjacencyGraphBase| class definition and private data members
@{template<class KeyType, class DataType>
class AdjacencyGraphBase :
public GraphInterface<KeyType,DataType> {
private:
  typedef KeyType Scalar;
  typedef aEdge<Scalar> EdgeType;
  typedef list<EdgeType> SublistType;
  typedef vector<SublistType> AdjType;

  AdjType          adjList;
@}

As in \verb|MatrixGraphBase|, a set of constructors and destructors analogous to
those given for the base classes are provided

@d \verb|AdjacencyGraphBase| constructors and destructors
@{public:
  explicit AdjacencyGraphBase() :
    GraphInterface() {}

  explicit AdjacencyGraphBase(bool isDi) :
    GraphInterface(isDi) {  }

  AdjacencyGraphBase ( const AdjacencyGraphBase& g ) :
    GraphInterface(g), adjList(b.adjList) {}

  ~AdjacencyGraphBase() {  }
@}

A few more methods are provided to assure compliance with the STL \verb|Container| concept:

@d Container methods for \verb|AdjacencyGraphBase|
@{  AdjacencyGraphBase& operator= ( const AdjacencyGraphBase& g ) {
    if ( &g == this )
      return *this;
    Graph::operator=(g);
    adjList = g.adjList;
    return *this;
  }

  virtual void swap ( AdjacencyGraphBase& g ) {
    Graph::swap(g);
    adjList.swap(g.adjList);
  }
@}

Now to the heart of the matter---the implementation of the interface methods required by \verb|graph_interface|. Some methods' code is identical to that for the matrix 
implementation; refer to that section for the code.

@d \verb|AdjacencyGraphBase| interface methods
@{@<Adjacency implementation of \verb|adjacentNodesOut|@>
@<Adjacency implementation of \verb|adjacentNodesIn|@>
@<Adjacency implementation of \verb|isConnected|@>
@<Adjacency implementation of \verb|findWeight|@>
@<Matrix implementation of \verb|addWeights|@>
@<Matrix implementation of \verb|removeOrphanNodes|@>
@<Adjacency implementation of \verb|addNode|@>
@<Matrix implementation of \verb|addNodes|@>
@<Adjacency implementation of \verb|removeNode|@>
@<Matrix implementation of \verb|removeNodes|@>
@<Adjacency implementation of \verb|removeAllNodes|@>
@<Adjacency implementation of \verb|addEdge|@>
@<Matrix implementation of \verb|addEdges|@>
@<Adjacency implementation of \verb|removeEdge|@>
@<Adjacency implementation of \verb|getEdges|@>
@<Adjacency implementation of \verb|numEdges|@>
@<Adjacency implementation of \verb|listAll|@>
@}

@d Adjacency implementation of \verb|adjacentNodesOut|
@{  template  <class OutputIterator>
  bool adjacentNodesOut(const DataType& nodeName,
                        OutputIterator o) const   {
     if( numEdges() == (Scalar)(0) )
	return false;

    if( !nodeExists(nodeName) )
      return false;

    Scalar index = getIndexOf(nodeName);

    const SublistType& sublist = adjList[index];

    if( sublist.empty() )
      return false;

    // you need to extract the index from the node
    // in this representation !!!!!!
    SublistType::const_iterator sublistIterator;
    for( sublistIterator = sublist.begin();
         sublistIterator != sublist.end(); ++sublistIterator )
	*o = nodeNameByIndex( sublistIterator->index() );

    return true;
  }
@}

@d Adjacency implementation of \verb|adjacentNodesIn|
@{  template  <class OutputIterator>
  bool adjacentNodesIn(const DataType& nodeName,
                       OutputIterator o) const   {
    if( !nodeExists(nodeName) )
      return false;
    bool has_any(false);
    Scalar index = getIndexOf(nodeName);

    NodeContainer::const_iterator nodeIterator =
      nodeContainer.begin();
    SublistType::const_iterator sublistIterator;

    for ( Scalar x(0); x < adjList.size(); ++x, ++nodeIterator)
      for( sublistIterator = adjList[x].begin();
           sublistIterator != adjList[x].end();
           ++sublistIterator )
        if( x != index )  // pass over already-examined sublist
          if( (*sublistIterator).index() == index )  
            {
              *o = (*nodeIterator).second;
              has_any = true;
              break;    // not a multigraph
            }
    return has_any;
  } 
@}

@d Adjacency implementation of \verb|isConnected|
@{  virtual bool isConnected(const DataType& nodeName) const  {
    if( !nodeExists(nodeName) )
      return false;   // redundant but doesn't hurt
    Scalar index = getIndexOf(nodeName);

    // ARE THERE EDGES COMING IN?
    SublistType::const_iterator sublistIterator;

    for ( Scalar x(0); x < adjList.size(); ++x )
      for( sublistIterator = adjList[x].begin();
           sublistIterator != adjList[x].end();
           ++sublistIterator )
        if( x != index )  // pass over already-examined sublist
          if( (*sublistIterator).index() == index )
            return true;

    // ARE THERE EDGES GOING OUT?
    if( !adjList[index].empty() )
      return true;

    return false;
  }
@}

@d Adjacency implementation of \verb|findWeight|
@{  virtual Scalar findWeight( const DataType& nodeName1,
                             const DataType& nodeName2 ) const  {
    if (!nodeExists(nodeName1) | !nodeExists(nodeName2))
      return Scalar(0);

    int index1 = getIndexOf(nodeName1);
    int index2 = getIndexOf(nodeName2);

    SublistType::const_iterator i;
    for( i = adjList[index1].begin();
         i != adjList[index1].end(); ++i )
      if( (*i).index() == index2 )
        return (*i).getWeight();

    return Scalar(0); // safety valve
  }
@}

@d Adjacency implementation of \verb|addNode|
@{  virtual void addNode ( const DataType& nodeName )   {
    Graph::addNode(nodeName);   // adds to nodeContainer
    int index = getIndexOf( nodeName );
    adjList.insert( adjList.begin() + getIndexOf( nodeName ),
                    SublistType() );

    for( AdjType::iterator i = adjList.begin();
         i < adjList.end();
         ++i )
      for( SublistType::iterator j = (*i).begin();
           j != (*i).end();
           ++j )
      {
        if( (*j).index() >= index )
          (*j).incIndex();
      }
  }
@}

@d Adjacency implementation of \verb|removeNode|
@{  virtual void removeNode ( const DataType& nodeName )  {
    if( !nodeExists( nodeName ) )
      return;

    int index = getIndexOf( nodeName );

    for( int h = 0; h < adjList.size(); ++h )
      removeEdge( nodeNameByIndex(h), nodeName );

    Graph::removeNode(nodeName);
    adjList.erase( adjList.begin() + index );

    for( AdjType::iterator i = adjList.begin();
         i < adjList.end();
         ++i )
      for( SublistType::iterator j = (*i).begin();
           j != (*i).end();
           ++j )
      {
        if( (*j).index() > index )
          (*j).decIndex();
      }
  }
@} 

@d Adjacency implementation of \verb|removeAllNodes|
@{  virtual void removeAllNodes()  {
    Graph::removeAllNodes();
    adjList.clear();
  }
@}

@d Adjacency implementation of \verb|addEdge|
@{  virtual bool addEdge(const DataType& nodeName1,
                       const DataType& nodeName2,
                       const Scalar weight=0)
  {
    if (!nodeExists(nodeName1) || !nodeExists(nodeName2))
      return false;

    // Make sure the adjacency list has been allocated
    // (if this is the first call to addEdge)
    if (adjList.empty())
      allocateGraph();

    Scalar index1 = getIndexOf(nodeName1);
    Scalar index2 = getIndexOf(nodeName2);

    adjList[index1].push_back( EdgeType( index2, weight ) );

    return true;
  }
@}

@d Adjacency implementation of \verb|removeEdge|
@{  virtual bool removeEdge(const DataType& nodeName1,
                          const DataType nodeName2)  {
    if (!nodeExists(nodeName1) || !nodeExists(nodeName2))
      return false;

    Scalar index1 = getIndexOf(nodeName1);
    Scalar index2 = getIndexOf(nodeName2);

    for( SublistType::iterator i = adjList[index1].begin();
         i != adjList[index1].end(); ++i )
      if( (*i).index() == index2 ) // found it! Assume no repeats
        {
          adjList[index1].erase(i);
          return true;
        }
    return false;   // not found in sublist
  }
@}

@d Adjacency implementation of \verb|getEdges|
@{  template <class OutputIterator>
    void getEdges ( OutputIterator o ) const {
    SublistType::const_iterator sublistIterator;
    for( Scalar x(0); x < adjList.size(); ++x)
      for( sublistIterator = adjList[x].begin();
           sublistIterator != adjList[x].end();
           ++sublistIterator )
        {
          pair<DataType, DataType> p(nodeNameByIndex(x),
              nodeNameByIndex( (*sublistIterator).index() ) );

          *o = p;
       }
  } 
@}

@d Adjacency implementation of \verb|numEdges|
@{  virtual Scalar numEdges() const  {
    Scalar count=0;

    SublistType::const_iterator sublistIterator;
    for( Scalar x(0); x < adjList.size(); ++x)
      for( sublistIterator = adjList[x].begin();
           sublistIterator != adjList[x].end();
           ++sublistIterator )
        {
          ++count;
        }

    return count;
  }
@}

@d Adjacency implementation of \verb|listAll|
@{  virtual void listAll(ostream& o) const {

    Graph::listAll(o);   // takes care of node listing

    o << endl << "Edge listings: ";
    if( adjList.size() ) o << endl;
    else o << "None." << endl;

    vector<pair<DataType, DataType> > edges;
    getEdges( back_inserter(edges) );
    for( vector<pair<DataType, DataType> >::const_iterator i =
      edges.begin();
      i < edges.end(); ++i )
        cout << (*i).first << " to " << (*i).second << endl;
  }
@}

A single graph allocation method is provided in the \verb|protected| section:

@d Protected methods for \verb|AdjacencyGraphBase|
@{  void allocateGraph()  {
    if (nodeContainer.size()>0)
      for (Scalar x(0); x < nodeContainer.size(); ++x)
        adjList.push_back( SublistType() );
  }

};  // END CLASS AdjacencyGraphBase       
@}

Finally, and output operator is provided:

@d External overloaded output operations for \verb|AdjacencyGraphBase|
@{template<class KeyType, class DataType>
ostream& operator<< (ostream& o,
                     const AdjacencyGraphBase<KeyType,DataType>& g)
{
  g.listAll(o);
  return o;
}
@}

%%% END ADJACENCYGRAPHBASE

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% adjacencygraph.h %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

The instantiable class \verb|adjacency_graph| parallels the
\verb|matrix_graph| class, providing functionality for compliance
with the \verb|Container| concept:

@o adjacencygraph.h -d
@{#ifndef ADJACENCY_GRAPH_H
#define ADJACENCY_GRAPH_H

@<\verb|adjacency_graph| directives and header files@>
@<\verb|adjacency_graph| class definition and type-definitions@>
@<Constructors for \verb|adjacency_graph|@>
@<\verb|adjacency_graph| container wrappers for STL compliance@>
#endif
@}


We begin with the usual directives and file-inclusion:

@d \verb|adjacency_graph| directives and header files
@{#include "adjacencygraphbase.h"
@}

Finally, to the instantiable class for this implementation:
\verb|adjacency_graph|.  Again, the class definition is very similar
to that of \verb|matrix_graph|, introducing members whose presence is
required for compliance with the STL \verb|Container| concept:

@d \verb|adjacency_graph| class definition and type-definitions
@{template <class KeyType, class DataType>
class adjacency_graph : public AdjacencyGraphBase<KeyType,DataType>
{
public:
  typedef DataType           value_type;
  typedef DataType&          reference_type;
  typedef const DataType&    const_reference;
  typedef DataType*          pointer;
  typedef ptrdiff_t          difference_type;
  typedef size_t             size_type;
  typedef graph_iterator
    <NodeContainer::iterator,DataType> iterator;
  typedef graph_iterator
    <NodeContainer::const_iterator,DataType> const_iterator;
  typedef graph_iterator
    <NodeContainer::reverse_iterator,DataType> reverse_iterator;
  typedef graph_iterator
    <NodeContainer::const_reverse_iterator,DataType>
                                          const_reverse_iterator;
@}

The constructors:

@d Constructors for \verb|adjacency_graph|
@{  explicit adjacency_graph( bool isDirected ) :
                        AdjacencyGraphBase(isDirected)
			{  }

  explicit adjacency_graph() : 
    AdjacencyGraphBase() {}

  adjacency_graph ( const adjacency_graph& g ) :
                        AdjacencyGraphBase(g) { }
@}

As with \verb|matrix_graph|, the following members are provided for
compliance with the STL \verb|Container| concept:

@d \verb|adjacency_graph| container wrappers for STL compliance
@{
  iterator begin() { return nodeContainer.begin(); }
  const_iterator begin() const 
    { return nodeContainer.begin(); }
  iterator end() { return nodeContainer.end(); }
  const_iterator end() const { return nodeContainer.end(); }
  reverse_iterator rbegin() { return nodeContainer.rbegin(); }
  const_reverse_iterator rbegin() const
    { return nodeContainer.rbegin(); }
  reverse_iterator rend() { return nodeContainer.rend(); }
  const_reverse_iterator rend() const
    { return nodeContainer.rend(); }
  bool empty() const { return nodeContainer.empty(); }
  void clear() { removeAllNodes(); deAllocateGraph(); return; }
  void swap ( adjacency_graph& g ) { GraphInterface::swap(g); }
  Scalar max_size() const { return nodeContainer.max_size(); }

  adjacency_graph& operator=( const adjacency_graph& g )
  {
    AdjacencyGraphBase::operator=(g);
    return *this;
  }
};
@}

\subsection{Adjacency Graph Test Application}

A program that demonstrates the functionality of
\verb|adjacency_graph| follows.  The tests are identical to those
performed for \verb|matrix_graph|.

@o main2.cc -d
@{
@<Header files for the main2 program@>
@<Declare an \verb|adjacency_graph| object@>
@<First test@>
@<Second test@>
@<Third test@>
@<Fourth test@>
@<Fifth test@>
@}

First, we include the header file for the \verb|adjacency_graph| class:

@d Header files for the main2 program
@{#include "adjacencygraph.h"
@}

Then, we declare the graph object we will use.

@d Declare an \verb|adjacency_graph| object
@{int main()
{
  adjacency_graph<unsigned int,string> g(true);
@}

See the code given for the \verb|matrix_graph| tests.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

\section{Appendix B: Algorithms}

Here, we present the source code for the three algorithms we have implemented 
in our distributed graph system.

The algorithms are : 
\begin{enumerate}
\item \verb|has_cycle|---tests whether a directed graph has a cycle.
\item \verb|topo_sort|---topologically sorts the nodes of a directed graph.
\item \verb|mst_prim|---finds the minimum spanning tree of a given graph.
\end{enumerate}

@o algorithms.h -d
@{#ifndef ALGORITHMS_H
#define ALGORITHMS_H

@<\verb|has_cycle|@>
@<\verb|topo_sort|@>
@<\verb|mst_prim|@>

#endif
@}

All of the below code was adapted for STL from pseudocode given in
\cite{Cormen}.

\subsection{Testing for a cycle}

Via a breadth-first-search of the graph, \verb|has_cycle| determines 
whether or not a directed graph has a cycle.

@d \verb|has_cycle|
@{  @<hc includes and directives@>
  @<hc class definition and private members@>

public:
  @<hc constructor@>
  @<hc run initialization@>
  @<hc run main loop@>
}; // end class has_cycle
@}

The algorithm requires a queue; an STL \verb|queue| is used. The
algorithm also requires node coloring. Three arbitrary constants are
defined to represent these "colors."

@d hc includes and directives
@{#include "adjacencygraph.h"
#include "matrixgraph.h"
#include<queue>

#define WHITE 0
#define GRAY 1
#define BLACK 2

using namespace disgraph;
@}

A pointer to a graph is maintained as a private member.  This pointer
will be assigned to the graph object on which the algorithm will
operate.

@d hc class definition and private members
@{template<class KeyType, class DataType>
class has_cycle {
private:

#ifdef MAT
typedef matrix_graph<NodeType, NodeType> GraphType;
#else
typedef adjacency_graph<NodeType, NodeType> GraphType;
#endif

  GraphType* G;
@}

@d hc constructor
@{
  has_cycle() {}
@}

All nodes are initially colored white (unvisited). The node at which
the breadth-first search begins (by default the node which has been
assigned an index of $0$ by the graph object) is then colored white
(visited).

@d hc run initialization
@{  bool run( GraphType& g ) {
    G = &g;

    int num_vertices = G->size();
    if( !num_vertices ) return false;

    int* color = new int[num_vertices];

    for( int i = 0; i < num_vertices; ++i )
      color[i] = WHITE;
    queue<DataType> Q;

    color[0] = GRAY;
    Q.push( 0 );

@}

The breadth-first search proceeds. Nodes that have been examined are
colored gray, and nodes that are "finished"---those that will no
longer participate in the search--- are colored black. If a black node
is found during the search, the graph has a cycle; \verb|true| is
immediately returned.

If the search has examined all nodes without encountering a node that
it has previously colored black, no cycles exist in the
graph---\verb|false| is therefore returned.

@d hc run main loop
@{    DataType u, v;
    while( !Q.empty() ) {
      u = Q.front();

      vector<DataType> neighbors;
      G->adjacentNodesOut( u, back_inserter(neighbors) );

      for( vector<DataType>::iterator p = neighbors.begin();
           p < neighbors.end(); ++p )
        {
          v = *p;
          if( color[G->getIndexOf(v)] == BLACK || v == u ) {
            delete color;
          return true;
        }
       if( color[G->getIndexOf(v)] == WHITE ) {
              color[G->getIndexOf(v)] = GRAY;
              Q.push(v);
          }
      }
      Q.pop();
      color[G->getIndexOf(u)] = BLACK;
    }
    delete color;
    return false;
  } // end run()
@}

\subsection{Topological Sort}

This implementation of the standard topological sort algorithm \cite{Cormen} 
uses depth-first search discovery times to determine the ordering of nodes 
to be returned. If the graph has a cycle, the ordering of returned nodes is 
simply an ordering of their depth-first search discovery times.
 
@d \verb|topo_sort|
@{  @<ts includes and directives@>
  @<ts private data members@>
  @<\verb|void DFS_Visit|@>
public:
  @<ts constructor@>
  @<ts \verb|run|@>
}; // end class topo_sort
@}

This implementation requires only two "colors."

@d ts includes and directives
@{#include "adjacencygraph.h"
#include "matrixgraph.h"

#define WHITE 0
#define GRAY 1
@}

A pointer \verb|color| will be assigned to an array holding the
current colorings of the nodes. As they are discovered, nodes are
stored in \verb|returnNodes|.

@d ts private data members
@{template<class KeyType, class DataType>
class topo_sort {
private:
#ifdef MAT
typedef matrix_graph<NodeType, NodeType> GraphType;
#else
typedef adjacency_graph<NodeType, NodeType> GraphType;
#endif
  GraphType* G;

  int* color;
  vector<DataType> return_nodes;
@}

\verb|DFS_Visit| recursively visits the nodes of the graph and stores
discovered nodes in \verb|returnNodes|.

@d \verb|void DFS_Visit|
@{  void DFS_Visit( int u ) {
    color[G->getIndexOf(u)] = GRAY;

    vector<DataType> neighbors;

    G->adjacentNodesOut( u, back_inserter( neighbors ) );
    for( vector<DataType>::iterator i = neighbors.begin();
         i < neighbors.end(); ++i )
      if( color[G->getIndexOf(*i)] == WHITE )
        DFS_Visit( *i );

    return_nodes.push_back( u );
  }
@}

@d ts constructor
@{
  topo_sort() {}
@}

A node at which to begin the search must be provided when running this
algorithm.  The body of the below method simply performs some
bookkeeping, calls \verb|DFS_Visit|, and then supplies the discovered
nodes to the passed-in output iterator.

@d ts \verb|run|
@{  template<class OutputIterator>
  void run( GraphType& g, 
	const DataType& u, OutputIterator o ) {
    // uses DFS
    // u must exist in the graph
    G = &g;

    int num_vertices = G->size();
    if( !num_vertices )
      return;

    color = new int[num_vertices];
    return_nodes.clear();

    for( int i = 0; i < num_vertices; ++i )
      color[i] = WHITE;

    DFS_Visit( u );

    delete color;

    for( int i = return_nodes.size() - 1; i >= 0; --i )
      *o = return_nodes[i];
  }
@}

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\subsection{Finding a Minimum Spanning Tree}

The below code to find the minimum spanning tree of a graph is derived from 
Prim's algorithm as described in \cite{Cormen}.

@d \verb|mst_prim|
@{  @<mst includes and directives@>
  @<\verb|second_less| predicate@>

template<class KeyType, class DataType>
class mst_prim {
private:
  @<mst private data members@>
  @<mst \verb|init|@>
  @<mst \verb|load_return_graph|@>
  @<mst \verb|process_edge|@>

public:
  @<mst constructor@>
  @<mst \verb|run|@>
}; // end class mst_prim
@}

Some representation of "infinity" must be provided. Any negative
number will serve this purpose, assuming that all edge weights of the
graph are nonnegative.

@d mst includes and directives
@{#include "adjacencygraph.h"
#include "matrixgraph.h"

#define INFINITY -1
@}

Each node must be associated with a key. Here, this is done using a
\verb|pair| in which the first element is the node and the second is
its key.

The algorithm depends on the ordering of nodes based on their keys. 
Such an ordering is enabled by the below predicate which compares 
the \verb|second| member of a pair. Here this ordering must take our 
representation of infinity into account; any key with a value of 
\verb|INFINITY| is determined to be greater than any other key.

@d \verb|second_less| predicate
@{template<class Pair>
class second_less
{
public:
  typedef Pair first_argument_type;
  typedef Pair second_argument_type;
  typedef bool result_type;

  bool operator() (const Pair& p1, const Pair& p2) const {
    if( p2.second == INFINITY )
      return true;
    if( p1.second == INFINITY )
      return false;
    return (p1.second < p2.second);
  }
};
@}

Several private data structures are maintained for use during the
algorithm's execution.

@d mst private data members
@{#ifdef MAT
typedef matrix_graph<NodeType, NodeType> GraphType;
#else
typedef adjacency_graph<NodeType, NodeType> GraphType;
#endif

  GraphType* G;
  GraphType* return_graph;

  typedef pair<DataType, int> nodeWithKey;
  typedef pair<DataType, DataType> anEdge;
  typedef vector<nodeWithKey> HeapType;
  HeapType Q;

  vector<anEdge> returnEdges;
  vector<DataType> returnNodes;
@}

After ensuring that all private data structures have been cleared, \verb|init| 
assigns the key \verb|INFINITY| to all nodes except for the root node.

@d mst \verb|init|
@{  bool init( const int& r ) {
    return_graph = new GraphType( true );
    if( !(G->size()) || !(G->numEdges()) )
      return false;

    Q.erase( Q.begin(), Q.end() );
    returnNodes.erase( returnNodes.begin(), returnNodes.end() );
    returnEdges.erase( returnEdges.begin(), returnEdges.end() );

    GraphType::iterator i = G->begin();
    for( ; i != G->end(); ++i )
      if( *i == r )
        Q.push_back( nodeWithKey( *i, 0 ) );
      else
        Q.push_back( nodeWithKey(*i, INFINITY) );

    sort( Q.begin(), Q.end(), second_less<nodeWithKey>() );
    return true;
  }
@}

The graph instance to be returned must be populated with the data 
gathered throughout algorithm execution:

@d mst \verb|load_return_graph|
@{  void load_return_graph() {
    return_graph->addNodes( returnNodes.begin(), 
                            returnNodes.end() );

    for( int k = 0; k < returnEdges.size(); ++k )
      return_graph->addEdge( returnEdges[k].first,
                             returnEdges[k].second,
                             G->findWeight( returnEdges[k].first,
                                       returnEdges[k].second ) );
  }
@}

The majority of the processing done by this algorithm occurs in the below 
method.

@d mst \verb|process_edge|
@{  void process_edge( const DataType& u, const DataType& v ) {
    int key_of_v;
    HeapType::iterator it = Q.begin();

    for( ; it < Q.end(); it++ )
      if( (*it).first == v )  {
        key_of_v = (*it).second;
        break;
      }

    if( it != Q.end() )  {
      int uv_weight = 0; // so will work w/ directed ...
      if( !(uv_weight = G->findWeight( u, v )) )
        uv_weight = G->findWeight( v, u );

      if( uv_weight < key_of_v || key_of_v == INFINITY )  {
        // update edge list
        for( vector<anEdge>::iterator zz = returnEdges.begin();
             zz < returnEdges.end(); ++zz )
          if( (*zz).second == v )
            returnEdges.erase( zz );

        returnEdges.push_back( pair<DataType, DataType>( u, v ) );
        (*it).second = uv_weight;
      }
    }
  }
@}

@d mst constructor
@{
  mst_prim() {}
@}

The caller must provide a root node for the minimum spanning tree.

@d mst \verb|run|
@{  // r is the root of the MST-tree
  GraphType& run( GraphType& g, const DataType& r ) {
    G = &g;

    if( !init( r ) )
      return *return_graph;

    DataType u, v;
    int w;    // given weight (again, scalar assumed)

    while( Q.size() > 0 )  {
      u = (*(Q.begin())).first;
      Q.erase( Q.begin() );  // u = EXTRACT-MIN(Q)

      returnNodes.push_back( u );   // mark as already handled

      vector<DataType> neighbors;
      neighbors.erase( neighbors.begin(), neighbors.end() ); 
      G->adjacentNodesOut( u, back_inserter( neighbors ) );
      G->adjacentNodesIn( u, back_inserter( neighbors ) );

      for( vector<DataType>::iterator v = neighbors.begin();
           v < neighbors.end(); ++v )
        {
          process_edge( u, *v );
          sort( Q.begin(), Q.end(), second_less<nodeWithKey>() );
        }
    } // end while

    load_return_graph();
    return *return_graph;
  } // end run()
@}

\newpage
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\begin{thebibliography}{MMM83}

\bibitem[Baker97]{Baker}Sean Baker. \emph{CORBA Distributed Objects}, 
Addison-Wesley, 1997.

\bibitem[Cormen90]{Cormen}T. Cormen, C. Lieserson, R. Rivest. 
\emph{Introduction to Algorithms}, MIT Press, 1990.

\bibitem[Gamma95]{Gamma}E. Gamma, R. Helm, R. Johnson, J. Vlissides. 
\emph{Design Patterns : Elements of Reusable Object-Oriented Software}, 
Addison-Wesley, 1995.

\bibitem[GTL99]{GTL}GTL : Graph Template Library. 
\emph{http://www.fmi.uni-passau.de/~himsolt/Graphlet/GTL/}.

\bibitem[Knuth93]{Knuth}D. E. Knuth. 
\emph{The Stanford GraphBase : A Platform for Combinatorial Computing},
Addison Wesley, 1993.

\bibitem[MICO]{MICO}MICO : MICO Is CORBA. 
\emph{http://www.mico.org}.

\bibitem[Musser96]{Musser}D. Musser, A. Saini. 
\emph{STL Tutorial and Reference Guide : C++ Programming with the 
Standard Template Library}, Addison-Wesley, 1996.

\bibitem[OMG]{OMG}The Object Management Group. 
\emph{http://www.omg.org}.

\bibitem[Patterson96]{Patterson}D. Patterson, J. Hennessy. \emph{Computer Architecture: A Quantitative Approach}, 2nd Edition, Morgan Kaufmann, 1996.

\bibitem[Roemer99]{Roemer}K. Roemer, A. Puder. \emph{MICO 2.2 Implementation Manual (Version 2.2.5)}, 
available at \emph{http://www.mico.org}.

\bibitem[Roemer99b]{Roemer:mail}K. Roemer. From an email discussion.

\bibitem[Stroustrup97]{Stroustrup}B. Stroustrup. \emph{The C++ Programming Language}, 
3rd Edition, Addison-Wesley, 1997.

\end{thebibliography}

\end{document}
