
using namespace std;

#include <cstdlib>
#include <assert.h>
#include <fstream>
#include <sys/time.h>
#include <string.h>
#include <algorithm>
#include <vector>
#include <math.h> 
#include <omp.h>

#define NUM_P 3
#define RANDOM 0

double treelet_weights[] = {1.0, 1.0, 2.0, 2.0, 2.0, 2.0, 2.0, 5.0, 4.0, 3.0, 5.0, 5.0, 4.0, 3.0, 3.0, 4.0, 7.0, 7.0, 4.0, 7.0, 7.0, 7.0, 6.0, 6.0, 5.0, 5.0, 6.0, 5.0, 5.0, 4.0, 6.0, 3.0, 6.0, 4.0, 4.0, 5.0, 5.0, 5.0, 8.0, 9.0, 8.0, 5.0, 9.0, 9.0, 10.0, 9.0, 6.0, 10.0, 8.0, 6.0, 7.0, 8.0, 10.0, 5.0, 10.0, 11.0, 12.0, 10.0, 7.0, 6.0, 7.0, 6.0, 4.0, 6.0, 10.0, 11.0, 7.0, 10.0, 10.0, 6.0, 8.0, 9.0, 10.0, 10.0, 10.0, 7.0, 5.0, 9.0, 10.0, 9.0, 7.0, 9.0, 6.0};

double graphlet_weights[] = {1.0, 2.0, 2.0, 2.0, 3.0, 4.0, 3.0, 3.0, 4.0, 3.0, 4.0, 4.0, 4.0, 4.0, 3.0, 4.0, 6.0, 5.0, 4.0, 5.0, 6.0, 6.0, 4.0, 4.0, 4.0, 5.0, 7.0, 4.0, 6.0, 6.0, 7.0, 4.0, 6.0, 6.0, 6.0, 5.0, 6.0, 7.0, 7.0, 5.0, 7.0, 6.0, 7.0, 6.0, 5.0, 5.0, 6.0, 8.0, 7.0, 6.0, 6.0, 8.0, 6.0, 9.0, 5.0, 6.0, 4.0, 6.0, 6.0, 7.0, 8.0, 6.0, 6.0, 8.0, 7.0, 6.0, 7.0, 7.0, 8.0, 5.0, 6.0, 6.0, 4.0};
int num_treelets = 83;
int num_graphlets = 73;
double* weights;
int num_graphs;
double weights_sum;


double timer() 
{
  struct timeval tp;
  gettimeofday(&tp, NULL);
  return ((double) (tp.tv_sec) + 1e-6 * tp.tv_usec);
}

typedef struct graph {
  int n;
  unsigned m;
  int* out_array;
  unsigned* out_degree_list;
} graph;
#define out_degree(g, n) (g.out_degree_list[n+1] - g.out_degree_list[n])
#define out_vertices(g, n) &g.out_array[g.out_degree_list[n]]

#include "align_compute.cpp"
#include "align_score.cpp"
#include "align_init.cpp"


void create_csr(int n, unsigned m, int* srcs, int* dsts,
                int*& out_array, unsigned*& out_degree_list)
{  
  out_array = new int[m];
  out_degree_list = new unsigned int[n+1];

  for (unsigned i = 0; i < m; ++i)
    out_array[i] = 0;
  for (int i = 0; i < n+1; ++i)
    out_degree_list[i] = 0;

  unsigned* temp_counts = new unsigned[n];
  for (int i = 0; i < n; ++i)
    temp_counts[i] = 0;
  for (unsigned i = 0; i < m; ++i)
    ++temp_counts[srcs[i]];
  for (int i = 0; i < n; ++i)
    out_degree_list[i+1] = out_degree_list[i] + temp_counts[i];
  copy(out_degree_list, out_degree_list + n, temp_counts);
  for (unsigned i = 0; i < m; ++i)
    out_array[temp_counts[srcs[i]]++] = dsts[i];
  delete [] temp_counts;
}

void read_graph(char* filename, 
  int& n, unsigned& m, 
  int*& srcs, int*& dsts)
{
  ifstream file_m;
  string line;  
  file_m.open(filename);

  getline(file_m, line);
  n = atoi(line.c_str());
  getline(file_m, line);
  m = strtoul(line.c_str(), NULL, 0);
  m *= 2;
  
  int src, dst;
  unsigned int counter = 0;
 
  srcs = new int[m];
  dsts = new int[m];
  for (unsigned i = 0; i < m/2; ++i)
  {
    getline(file_m, line, ' ');   
    src = atoi(line.c_str());
    getline(file_m, line);
    dst = atoi(line.c_str());

    srcs[counter] = src;
    dsts[counter] = dst;
    ++counter;
    srcs[counter] = dst;
    dsts[counter] = src;
    ++counter;
  }
  assert(counter == m);

  file_m.close();
}

void read_in_counts(char* countsfile, long** counts, graph& g)
{
  ifstream file_in;
  string line;  
  file_in.open(countsfile);

  for (int i = 0; i < g.n; ++i)
  {
    getline(file_in, line, ' ');
    int cur_vert = atol(line.c_str());
    for (int j = 0; j < num_graphs-1; ++j)
    {
      getline(file_in, line, ' ');
      counts[cur_vert][j] = atol(line.c_str());
      assert(counts[cur_vert][j] >= 0);
    }
    getline(file_in, line);
    counts[cur_vert][num_graphs-1] = atol(line.c_str());
  }
}

int main(int argc, char** argv)
{
  setbuf(stdout, NULL);
  srand(time(0));
  if (argc < 6)
  {
    printf("%s [alpha] [graph1] [graph2] [counts1] [counts2] [t/g]\n", argv[0]);
    exit(0);
  }
  double alpha = atof(argv[1]);
  char* graph1 = argv[2];
  char* graph2 = argv[3];
  char* countsfile1 = argv[4];
  char* countsfile2 = argv[5];
  if (argv[6][0] == 'g')
  {
    weights = graphlet_weights;
    num_graphs = num_graphlets;
  }
  else if (argv[6][0] == 't')
  {
    weights = treelet_weights;
    num_graphs = num_treelets;
  }
  else
  {
    printf("need to supply (t)reelets or (g)raphlets as final argument");
    exit(0);
  }

  int* srcs1;
  int* srcs2;
  int* dsts1;
  int* dsts2;
  int n1;
  int n2;
  unsigned m1;
  unsigned m2;
  int* out_array1;
  int* out_array2;
  unsigned* out_degree_list1;
  unsigned* out_degree_list2;

  printf("Creating graphs %s %s ... ", graph1, graph2);
  double elt = timer();
  double start_time = elt;
  read_graph(graph1, n1, m1, srcs1, dsts1);
  read_graph(graph2, n2, m2, srcs2, dsts2);
  if (n1 > n2)
  {
    printf("First graph vertex count must be less than second\n");
    exit(0);
  }
  create_csr(n1, m1, srcs1, dsts1, out_array1, out_degree_list1);
  create_csr(n2, m2, srcs2, dsts2, out_array2, out_degree_list2);
  delete [] srcs1;
  delete [] srcs2;
  delete [] dsts1;
  delete [] dsts2;
  graph g1 = {n1, m1, out_array1, out_degree_list1};
  graph g2 = {n2, m2, out_array2, out_degree_list2};
  elt = timer() - elt;
  printf("done creating, %9.6lf\n", elt);

  printf("Reading counts %s %s ... ", countsfile1, countsfile2);
  elt = timer();
  long** counts1 = new long*[g1.n];
  long** counts2 = new long*[g2.n];
  for (int i = 0; i < g1.n; ++i) counts1[i] = new long[num_graphs];
  for (int i = 0; i < g2.n; ++i) counts2[i] = new long[num_graphs];
  read_in_counts(countsfile1, counts1, g1);
  read_in_counts(countsfile2, counts2, g2);
  elt = timer() - elt;
  printf("done reading, %9.6lf\n", elt);

  printf("Creating cost matrix ... ");
  elt = timer();
  float** costs = new float*[g1.n];
  for (int i = 0; i < g1.n; ++i) costs[i] = new float[g2.n];
  create_cost_matrix(g1, g2, counts1, counts2, costs, alpha);
  delete [] counts1;
  delete [] counts2;
  elt = timer() - elt;
  printf("done creating, %9.6lf\n", elt);

  printf("Creating power graphs ... ");
  elt = timer();
  graph* g1p = new graph[NUM_P]; g1p[0] = g1;
  graph* g2p = new graph[NUM_P]; g2p[0] = g2;
  printf(" 1 ");
  for (int i = 1; i < NUM_P; ++i)
  {
    g1p[i] = create_csr_power(g1, i+1);
    g2p[i] = create_csr_power(g2, i+1);
    printf(" %d ", i+1);
  }
  elt = timer() - elt;
  printf("done creating, %9.6lf\n", elt);

  printf("Doing alignment ... ");
  elt = timer();
  pair<int, int>* alignment = compute_alignment(g1p, g2p, costs);
  elt = timer() - elt;
  printf("done aligning, %9.6lf\n", elt);

  printf("Scoring ... ");
  elt = timer();
  double ec = edge_correctness(g1, g2, alignment);
  elt = timer() - elt;
  printf("done aligning, %9.6lf\n", elt);
  printf("EC: %9.6lf\n", ec);

  start_time = timer() - start_time;
  printf("Done Total, %9.6lf\n", start_time);
}
