#include <iostream>
#include <fstream>
#include <sstream>
#include <iomanip>
#include <cstring>

#include "mesh.h"
#include "edge.h"
#include "vertex.h"
#include "triangle.h"
#include "meshdata.h"

// to give a unique id number to each triangles
int Triangle::next_triangle_id = 0;


// =======================================================================
// MESH DESTRUCTOR 
// =======================================================================

Mesh::~Mesh() {
  // delete all the triangles
  std::vector<Triangle*> todo;
  for (triangleshashtype::iterator iter = triangles.begin();
       iter != triangles.end(); iter++) {
    Triangle *t = iter->second;
    todo.push_back(t);
  }
  int num_triangles = todo.size();
  for (int i = 0; i < num_triangles; i++) {
    removeTriangle(todo[i]);
  }
  // delete all the vertices
  int num_vertices = numVertices();
  for (int i = 0; i < num_vertices; i++) {
    delete vertices[i];
  }
}

// =======================================================================
// MODIFIERS:   ADD & REMOVE
// =======================================================================

Vertex* Mesh::addVertex(const Vec3f &position) {
  int index = numVertices();
  Vertex *v = new Vertex(index, position);
  vertices.push_back(v);
  if (numVertices() == 1)
    bbox = BoundingBox(position,position);
  else 
    bbox.Extend(position);
  return v;
}


void Mesh::addTriangle(Vertex *a, Vertex *b, Vertex *c) {
  // create the triangle
  Triangle *t = new Triangle();
  // create the edges
  Edge *ea = new Edge(a,b,t);
  Edge *eb = new Edge(b,c,t);
  Edge *ec = new Edge(c,a,t);
  // point the triangle to one of its edges
  t->setEdge(ea);
  // connect the edges to each other
  ea->setNext(eb);
  eb->setNext(ec);
  ec->setNext(ea);
  // verify these edges aren't already in the mesh 
  // (which would be a bug, or a non-manifold mesh)
  assert (edges.find(std::make_pair(a,b)) == edges.end());
  assert (edges.find(std::make_pair(b,c)) == edges.end());
  assert (edges.find(std::make_pair(c,a)) == edges.end());
  // add the edges to the master list
  edges[std::make_pair(a,b)] = ea;
  edges[std::make_pair(b,c)] = eb;
  edges[std::make_pair(c,a)] = ec;
  // connect up with opposite edges (if they exist)
  edgeshashtype::iterator ea_op = edges.find(std::make_pair(b,a)); 
  edgeshashtype::iterator eb_op = edges.find(std::make_pair(c,b)); 
  edgeshashtype::iterator ec_op = edges.find(std::make_pair(a,c)); 
  if (ea_op != edges.end()) { ea_op->second->setOpposite(ea); }
  if (eb_op != edges.end()) { eb_op->second->setOpposite(eb); }
  if (ec_op != edges.end()) { ec_op->second->setOpposite(ec); }
  // add the triangle to the master list
  assert (triangles.find(t->getID()) == triangles.end());
  triangles[t->getID()] = t;
}


void Mesh::removeTriangle(Triangle *t) {
  Edge *ea = t->getEdge();
  Edge *eb = ea->getNext();
  Edge *ec = eb->getNext();
  Vertex *a = ea->getStartVertex();
  Vertex *b = eb->getStartVertex();
  Vertex *c = ec->getStartVertex();
  // remove these elements from master lists
  edges.erase(std::make_pair(a,b)); 
  edges.erase(std::make_pair(b,c)); 
  edges.erase(std::make_pair(c,a)); 
  triangles.erase(t->getID());
  // clean up memory
  delete ea;
  delete eb;
  delete ec;
  delete t;
}


// =======================================================================
// Helper functions for accessing data in the hash table
// =======================================================================

Edge* Mesh::getMeshEdge(Vertex *a, Vertex *b) const {
  edgeshashtype::const_iterator iter = edges.find(std::make_pair(a,b));
  if (iter == edges.end()) return NULL;
  return iter->second;
}

Vertex* Mesh::getChildVertex(Vertex *p1, Vertex *p2) const {
  vphashtype::const_iterator iter = vertex_parents.find(std::make_pair(p1,p2)); 
  if (iter == vertex_parents.end()) return NULL;
  return iter->second; 
}

void Mesh::setParentsChild(Vertex *p1, Vertex *p2, Vertex *child) {
  assert (vertex_parents.find(std::make_pair(p1,p2)) == vertex_parents.end());
  vertex_parents[std::make_pair(p1,p2)] = child; 
}


// =======================================================================
// the load function parses very simple .obj files
// the basic format has been extended to allow the specification 
// of crease weights on the edges.
// =======================================================================

#define MAX_CHAR_PER_LINE 200

void Mesh::Load(const std::string &input_file) {

  std::ifstream istr(input_file.c_str());
  if (!istr) {
    std::cout << "ERROR! CANNOT OPEN: " << input_file << std::endl;
    return;
  }

  char line[MAX_CHAR_PER_LINE];
  std::string token, token2;
  float x,y,z;
  int a,b,c;
  int index = 0;
  int vert_count = 0;
  int vert_index = 1;

  // read in each line of the file
  while (istr.getline(line,MAX_CHAR_PER_LINE)) { 
    // put the line into a stringstream for parsing
    std::stringstream ss;
    ss << line;

    // check for blank line
    token = "";   
    ss >> token;
    if (token == "") continue;

    if (token == std::string("usemtl") ||
	token == std::string("g")) {
      vert_index = 1; 
      index++;
    } else if (token == std::string("v")) {
      vert_count++;
      ss >> x >> y >> z;
      addVertex(Vec3f(x,y,z));
    } else if (token == std::string("f")) {
      a = b = c = -1;
      ss >> a >> b;
      // handle faces with > 3 vertices
      // assume the face can be triangulated with a triangle fan
      while (ss >> c) {
        int a_ = a-vert_index;
        int b_ = b-vert_index;
        int c_ = c-vert_index;
        assert (a_ >= 0 && a_ < numVertices());
        assert (b_ >= 0 && b_ < numVertices());
        assert (c_ >= 0 && c_ < numVertices());
        addTriangle(getVertex(a_),getVertex(b_),getVertex(c_));
        b = c;
      }
    } else if (token == std::string("e")) {
      a = b = -1;
      ss >> a >> b >> token2;
      // whoops: inconsistent file format, don't subtract 1
      assert (a >= 0 && a <= numVertices());
      assert (b >= 0 && b <= numVertices());
      if (token2 == std::string("inf")) x = 1000000; // this is close to infinity...
      x = atof(token2.c_str());
      Vertex *va = getVertex(a);
      Vertex *vb = getVertex(b);
      Edge *ab = getMeshEdge(va,vb);
      Edge *ba = getMeshEdge(vb,va);
      assert (ab != NULL);
      assert (ba != NULL);
      ab->setCrease(x);
      ba->setCrease(x);
    } else if (token == std::string("vt")) {
    } else if (token == std::string("vn")) {
    } else if (token[0] == '#') {
    } else {
      printf ("LINE: '%s'",line);
    }
  }

  std::cout << "Loaded " << numTriangles() << " triangles." << std::endl;

  assert (numTriangles() > 0);
  num_mini_triangles = 0;
}


// =======================================================================
// DRAWING
// =======================================================================

Vec3f ComputeNormal(const Vec3f &p1, const Vec3f &p2, const Vec3f &p3) {
  Vec3f v12 = p2;
  v12 -= p1;
  Vec3f v23 = p3;
  v23 -= p2;
  Vec3f normal;
  Vec3f::Cross3(normal,v12,v23);
  normal.Normalize();
  return normal;
}


// boundary edges are red, crease edges are yellow
Vec3f EdgeColor(Edge *e) {
  if (GLOBAL_args->mesh_data->wireframe) {
    if (e->getOpposite() == NULL) {
      return Vec3f(1,0,0); 
    } else if (e->getCrease() > 0) {
      return Vec3f(1,1,0);
    } else {
      return Vec3f(0,0,0.0);
    }
  } else {
    return Vec3f(1,1,1);
  }
}


// =================================================================
// SUBDIVISION
// =================================================================


void Mesh::LoopSubdivision() {
  printf ("Subdivide the mesh!\n");
  
  // =====================================
  // ASSIGNMENT: complete this functionality
  // =====================================

}


// =================================================================
// SIMPLIFICATION
// =================================================================

void Mesh::Simplification(int target_tri_count) {
  // clear out any previous relationships between vertices
  vertex_parents.clear();

  printf ("Simplify the mesh! %d -> %d\n", numTriangles(), target_tri_count);

  // =====================================
  // ASSIGNMENT: complete this functionality
  // =====================================
}


// =================================================================
// PACK MESH -- PREPARE FOR RENDERING
// =================================================================

void Mesh::packMesh(MeshData *mesh_data) {

  // clean up the old data (if any)
  delete [] mesh_data->triData;
  
  /*
  // To create a wireframe rendering...
  // Each mesh triangle is actually rendered as 3 small triangles
  //           b
  //          /|\
  //         / | \
  //        /  |  \
  //       /   |   \   
  //      /    |    \  
  //     /    .'.    \   
  //    /  .'     '.  \  
  //   /.'           '.\ 
  //  a-----------------c
  //
  */

  // we triple the number of triangles if wireframe is active
  mesh_data->triCount = GLOBAL_args->mesh->numTriangles();
  if (GLOBAL_args->mesh_data->wireframe) { mesh_data->triCount *= 3; }

  // allocate space for the new data
  mesh_data->triData = new float[12*3* mesh_data->triCount];


  // Loop over all of the triangles
  float* current = mesh_data->triData;
  int which = 0;

  for (triangleshashtype::iterator iter = GLOBAL_args->mesh->triangles.begin();
       iter != GLOBAL_args->mesh->triangles.end(); iter++) {

    Triangle *t = iter->second;

    Vec3f a = (*t)[0]->getPos();
    Vec3f b = (*t)[1]->getPos();
    Vec3f c = (*t)[2]->getPos();
    // use simple averaging to find centroid & average normal
    Vec3f centroid = 1.0f / 3.0f * (a+b+c);

    // for flat shading
    Vec3f normal = ComputeNormal(a,b,c);
    Vec3f na = normal;
    Vec3f nb = normal;
    Vec3f nc = normal;
    
    // for smooth gouraud shading
    if (GLOBAL_args->mesh_data->gouraud) {


      // =====================================
      // ASSIGNMENT: complete this functionality
      // =====================================


    }
    
    // determine edge colors (when wireframe is enabled)
    // the center is white, the colors of the two vertices depend on
    // whether the edge is a boundary edge (red) or crease edge (yellow)
    Vec3f edgecolor_ab = EdgeColor(t->getEdge());
    Vec3f edgecolor_bc = EdgeColor(t->getEdge()->getNext());
    Vec3f edgecolor_ca = EdgeColor(t->getEdge()->getNext()->getNext());
  
    if (!GLOBAL_args->mesh_data->wireframe) {
      
      // Just draw the one triangle if wireframe is not active
      float12 ta = { float(a.x()),float(a.y()),float(a.z()),1, float(na.x()),float(na.y()),float(na.z()),0, 1.0,1.0,1.0,1 };
      float12 tb = { float(b.x()),float(b.y()),float(b.z()),1, float(nb.x()),float(nb.y()),float(nb.z()),0, 1.0,1.0,1.0,1 };
      float12 tc = { float(c.x()),float(c.y()),float(c.z()),1, float(nc.x()),float(nc.y()),float(nc.z()),0, 1.0,1.0,1.0,1 };
      memcpy(current, &ta, sizeof(float)*12); current += 12; which++;
      memcpy(current, &tb, sizeof(float)*12); current += 12; which++;
      memcpy(current, &tc, sizeof(float)*12); current += 12; which++;
      
    } else {

      // Draw 3 triangles if wireframe is active
      float12 ta = { float(a.x()),float(a.y()),float(a.z()),1, float(na.x()),float(na.y()),float(na.z()),0, float(edgecolor_ab.r()),float(edgecolor_ab.g()),float(edgecolor_ab.b()),1 };
      float12 tb = { float(b.x()),float(b.y()),float(b.z()),1, float(nb.x()),float(nb.y()),float(nb.z()),0, float(edgecolor_ab.r()),float(edgecolor_ab.g()),float(edgecolor_ab.b()),1 };
      float12 tc = { float(centroid.x()),float(centroid.y()),float(centroid.z()),1, float(normal.x()),float(normal.y()),float(normal.z()),0, 1.0,1.0,1.0,1 };
      memcpy(current, &ta, sizeof(float)*12); current += 12; which++;
      memcpy(current, &tb, sizeof(float)*12); current += 12; which++;
      memcpy(current, &tc, sizeof(float)*12); current += 12; which++;
      
      ta = { float(b.x()),float(b.y()),float(b.z()),1, float(nb.x()),float(nb.y()),float(nb.z()),0, float(edgecolor_bc.r()),float(edgecolor_bc.g()),float(edgecolor_bc.b()),1 };
      tb = { float(c.x()),float(c.y()),float(c.z()),1, float(nc.x()),float(nc.y()),float(nc.z()),0, float(edgecolor_bc.r()),float(edgecolor_bc.g()),float(edgecolor_bc.b()),1 };
      tc = { float(centroid.x()),float(centroid.y()),float(centroid.z()),1, float(normal.x()),float(normal.y()),float(normal.z()),0, 1.0,1.0,1.0,1 };
      memcpy(current, &ta, sizeof(float)*12); current += 12; which++;
      memcpy(current, &tb, sizeof(float)*12); current += 12; which++;
      memcpy(current, &tc, sizeof(float)*12); current += 12; which++;
      
      ta = { float(c.x()),float(c.y()),float(c.z()),1, float(nc.x()),float(nc.y()),float(nc.z()),0, float(edgecolor_ca.r()),float(edgecolor_ca.g()),float(edgecolor_ca.b()),1 };
      tb = { float(a.x()),float(a.y()),float(a.z()),1, float(na.x()),float(na.y()),float(na.z()),0, float(edgecolor_ca.r()),float(edgecolor_ca.g()),float(edgecolor_ca.b()),1 };
      tc = { float(centroid.x()),float(centroid.y()),float(centroid.z()),1, float(normal.x()),float(normal.y()),float(normal.z()),0, 1.0,1.0,1.0,1 };
      memcpy(current, &ta, sizeof(float)*12); current += 12; which++;
      memcpy(current, &tb, sizeof(float)*12); current += 12; which++;
      memcpy(current, &tc, sizeof(float)*12); current += 12; which++;
    }
  }

  // the boundingbox center and size will be used to adjust the camera
  Vec3f center;
  bbox.getCenter(center);
  
  mesh_data->bb_center.data[0] = center.x();
  mesh_data->bb_center.data[1] = center.y();
  mesh_data->bb_center.data[2] = center.z();
  
  mesh_data->bb_scale = 1.8 / float(bbox.maxDim());
}


// =================================================================
